<?php

namespace App\Filament\Resources\ProductPrices\Pages;

use App\Filament\Resources\ProductPrices\ProductPricesResource;
use App\Models\Product;
use App\Models\ProductPrice;
use App\Models\ProductVariant;
use App\Models\StockMovement;
use Filament\Actions\Action;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\Page;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Schema;

class UpdateProductPrices extends Page implements HasForms
{
    use InteractsWithForms;

    protected static string $resource = ProductPricesResource::class;

    protected string $view = 'filament.pages.product.product-prices';

    public ?array $data = [];

    public function mount(): void
    {
        $this->form->fill([
            'retail_margin' => 0,
            'wholesale_margin' => 0,
            'agent_margin' => 0,
        ]);
    }

    public function form(Schema $form): Schema
    {
        return $form
            ->statePath('data')
            ->schema([
                Grid::make(12)->schema([
                    Select::make('product_id')
                        ->label('المنتج')
                        ->options(Product::query()->orderBy('id', 'desc')->pluck('name', 'id'))
                        ->searchable()
                        ->preload()
                        ->required()
                        ->live()
                        ->afterStateUpdated(fn () => $this->data['product_variant_id'] = null)
                        ->columnSpan(['default' => 12, 'md' => 6]),

                    Select::make('product_variant_id')
                        ->label('متغير محدد (اختياري)')
                        ->options(function () {
                            $pid = (int) ($this->data['product_id'] ?? 0);
                            if ($pid <= 0) {
                                return [];
                            }

                            return ProductVariant::query()
                                ->where('product_id', $pid)
                                ->orderBy('id', 'desc')
                                ->get()
                                ->pluck('full_name', 'id')
                                ->toArray();
                        })
                        ->searchable()
                        ->preload()
                        ->disabled(fn () => (int) ($this->data['product_id'] ?? 0) <= 0)
                        ->columnSpan(['default' => 12, 'md' => 6]),

                    TextInput::make('retail_margin')
                        ->label('هامش المفرد %')
                        ->default(0)
                        ->numeric()
                        ->minValue(0)
                        ->columnSpan(['default' => 12, 'md' => 4]),

                    TextInput::make('wholesale_margin')
                        ->label('هامش الجملة %')
                        ->default(0)
                        ->numeric()
                        ->minValue(0)
                        ->columnSpan(['default' => 12, 'md' => 4]),

                    TextInput::make('agent_margin')
                        ->label('هامش الوكيل %')
                        ->default(0)
                        ->numeric()
                        ->minValue(0)
                        ->columnSpan(['default' => 12, 'md' => 4]),
                ]),
            ]);
    }

    protected function getHeaderActions(): array
    {
        return [
            Action::make('apply')
                ->label('تطبيق تحديث الأسعار')
                ->icon('heroicon-o-check')
                ->requiresConfirmation()
                ->action(fn () => $this->applyBulkUpdate()),
        ];
    }

    private function applyBulkUpdate(): void
    {
        $pid = (int) ($this->data['product_id'] ?? 0);
        if ($pid <= 0) {
            Notification::make()->title('اختر المنتج')->warning()->send();

            return;
        }

        $variantId = (int) ($this->data['product_variant_id'] ?? 0);

        $rm = (float) ($this->data['retail_margin'] ?? 0);
        $wm = (float) ($this->data['wholesale_margin'] ?? 0);
        $am = (float) ($this->data['agent_margin'] ?? 0);

        $variants = ProductVariant::query()
            ->where('product_id', $pid)
            ->when($variantId > 0, fn ($q) => $q->whereKey($variantId))
            ->get();

        if ($variants->isEmpty()) {
            Notification::make()->title('لا توجد متغيرات لهذا المنتج')->warning()->send();

            return;
        }

        $updated = 0;

        foreach ($variants as $v) {
            $cost = (int) (StockMovement::query()
                ->where('product_variant_id', $v->id)
                ->orderByDesc('id')
                ->value('unit_cost') ?? 0);

            if ($cost <= 0) {
                $cost = (int) ($v->cost ?? 0);
            }

            $retailPrice = (int) round($cost + ($cost * $rm / 100));
            $wholesalePrice = (int) round($cost + ($cost * $wm / 100));
            $agentPrice = (int) round($cost + ($cost * $am / 100));

            ProductPrice::query()->updateOrCreate(
                [
                    'product_id' => $pid,
                    'product_variant_id' => $v->id,
                ],
                [
                    'retail_margin' => $rm,
                    'wholesale_margin' => $wm,
                    'agent_margin' => $am,
                    'retail_price' => $retailPrice,
                    'wholesale_price' => $wholesalePrice,
                    'agent_price' => $agentPrice,
                    'status' => true,
                ]
            );

            $updated++;
        }

        Notification::make()
            ->title("تم تحديث الأسعار: {$updated} متغير")
            ->success()
            ->send();
    }
}
