<?php

namespace App\Filament\Resources\ProductPrices\Pages;

use App\Filament\Resources\ProductPrices\ProductPricesResource;
use App\Models\ProductVariant;
use App\Models\StockMovement;
use Filament\Actions\Action;
use Filament\Actions\DeleteAction;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\EditRecord;

class EditProductPrices extends EditRecord
{
    protected static string $resource = ProductPricesResource::class;

    protected function mutateFormDataBeforeSave(array $data): array
    {
        $mode = $this->data['pricing_mode'] ?? 'margin';

        $variantId = (int) ($data['product_variant_id'] ?? 0);
        if ($variantId <= 0) {
            return $data;
        }

        // كلفة آخر حركة (fallback من cost)
        $cost = (int) (StockMovement::query()
            ->where('product_variant_id', $variantId)
            ->orderByDesc('id')
            ->value('unit_cost') ?? 0);

        if ($cost <= 0) {
            $cost = (int) (ProductVariant::query()->whereKey($variantId)->value('cost') ?? 0);
        }

        if ($mode === 'margin') {
            $rm = (float) ($data['retail_margin'] ?? 0);
            $wm = (float) ($data['wholesale_margin'] ?? 0);
            $am = (float) ($data['agent_margin'] ?? 0);

            $data['retail_price'] = (int) round($cost + ($cost * $rm / 100));
            $data['wholesale_price'] = (int) round($cost + ($cost * $wm / 100));
            $data['agent_price'] = (int) round($cost + ($cost * $am / 100));

            return $data;
        }

        // direct: (اختياري) نحسب الهوامش من الأسعار لضمان التزامن
        $rp = (int) ($data['retail_price'] ?? 0);
        $wp = (int) ($data['wholesale_price'] ?? 0);
        $ap = (int) ($data['agent_price'] ?? 0);

        $data['retail_margin'] = ($cost > 0 && $rp > 0) ? (float) number_format((($rp - $cost) / $cost) * 100, 2, '.', '') : 0;
        $data['wholesale_margin'] = ($cost > 0 && $wp > 0) ? (float) number_format((($wp - $cost) / $cost) * 100, 2, '.', '') : 0;
        $data['agent_margin'] = ($cost > 0 && $ap > 0) ? (float) number_format((($ap - $cost) / $cost) * 100, 2, '.', '') : 0;

        return $data;
    }

    protected function getHeaderActions(): array
    {
        return [
            Action::make('sync_cost')
                ->label('تحديث الكلفة من آخر حركة')
                ->icon('heroicon-o-arrow-path')
                ->action(function () {
                    $variantId = (int) ($this->data['product_variant_id'] ?? 0);

                    if ($variantId <= 0) {
                        Notification::make()->title('اختر متغيراً أولاً')->warning()->send();

                        return;
                    }

                    $cost = (int) (StockMovement::query()
                        ->where('product_variant_id', $variantId)
                        ->orderByDesc('id')
                        ->value('unit_cost') ?? 0);

                    if ($cost <= 0) {
                        $cost = (int) (ProductVariant::query()->whereKey($variantId)->value('cost') ?? 0);
                    }

                    $this->data['unit_cost_preview'] = $cost;

                    // لو margin: احسب الأسعار فوراً
                    if (($this->data['pricing_mode'] ?? 'margin') === 'margin') {
                        $rm = (float) ($this->data['retail_margin'] ?? 0);
                        $wm = (float) ($this->data['wholesale_margin'] ?? 0);
                        $am = (float) ($this->data['agent_margin'] ?? 0);

                        $this->data['retail_price'] = (int) round($cost + ($cost * $rm / 100));
                        $this->data['wholesale_price'] = (int) round($cost + ($cost * $wm / 100));
                        $this->data['agent_price'] = (int) round($cost + ($cost * $am / 100));
                    }

                    Notification::make()->title('تم تحديث الكلفة')->success()->send();
                }),

            Action::make('recalc_prices')
                ->label('إعادة احتساب الأسعار')
                ->icon('heroicon-o-calculator')
                ->action(function () {
                    $cost = (int) ($this->data['unit_cost_preview'] ?? 0);

                    if ($cost <= 0) {
                        Notification::make()->title('لا توجد كلفة لحساب الأسعار')->warning()->send();

                        return;
                    }

                    if (($this->data['pricing_mode'] ?? 'margin') !== 'margin') {
                        Notification::make()->title('وضع التسعير حالياً إدخال مباشر')->info()->send();

                        return;
                    }

                    $rm = (float) ($this->data['retail_margin'] ?? 0);
                    $wm = (float) ($this->data['wholesale_margin'] ?? 0);
                    $am = (float) ($this->data['agent_margin'] ?? 0);

                    $this->data['retail_price'] = (int) round($cost + ($cost * $rm / 100));
                    $this->data['wholesale_price'] = (int) round($cost + ($cost * $wm / 100));
                    $this->data['agent_price'] = (int) round($cost + ($cost * $am / 100));

                    Notification::make()->title('تم احتساب الأسعار')->success()->send();
                }),

            DeleteAction::make(),
        ];
    }
}
