<?php

namespace App\Filament\Resources\ProductModels\Schemas;

use App\Models\ProductModel;
use Filament\Forms\Components\Checkbox;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use Filament\Support\Icons\Heroicon;

class ProductModelForm
{
    public static function configure(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make('معلومات أساسية')
                    ->description('المعلومات الأساسية للموديل ثلاثي الأبعاد')
                    ->icon(Heroicon::OutlinedCube)
                    ->schema([
                        Grid::make(2)
                            ->schema([
                                Select::make('product_id')
                                    ->label('المنتج')
                                    ->relationship('product', 'name')
                                    ->searchable()
                                    ->preload()
                                    ->required()
                                    ->createOptionForm([
                                        TextInput::make('name')
                                            ->label('اسم المنتج')
                                            ->required(),
                                    ])
                                    ->helperText('اختر المنتج الذي سيرتبط بهذا الموديل')
                                    ->columnSpan(1),

                                TextInput::make('name')
                                    ->label('اسم الموديل')
                                    ->required()
                                    ->maxLength(255)
                                    ->placeholder('مثال: موديل جوارب رياضية')
                                    ->helperText('اسم واضح للموديل')
                                    ->columnSpan(1),

                                Textarea::make('description')
                                    ->label('الوصف')
                                    ->rows(3)
                                    ->maxLength(1000)
                                    ->placeholder('وصف مختصر للموديل...')
                                    ->columnSpan(2),
                            ]),
                    ]),

                Section::make('ملفات الموديل')
                    ->description('رفع ملف GLB وصورة المعاينة')
                    ->icon(Heroicon::OutlinedPhoto)
                    ->schema([
                        Grid::make(2)
                            ->schema([
                                SpatieMediaLibraryFileUpload::make('glb_file')
                                    ->label('ملف الموديل (GLB)')
                                    ->collection(ProductModel::MEDIA_COLLECTION_GLB)
                                    ->acceptedFileTypes(['model/gltf-binary', '.glb'])
                                    ->maxSize(10240) // 10MB
                                    ->required()
                                    ->helperText('ارفع ملف GLB (الحد الأقصى: 10MB)')
                                    ->downloadable()
                                    ->openable()
                                    ->deletable()
                                    ->columnSpan(1),

                                SpatieMediaLibraryFileUpload::make('preview_image')
                                    ->label('صورة المعاينة')
                                    ->collection(ProductModel::MEDIA_COLLECTION_PREVIEW)
                                    ->image()
                                    ->imageEditor()
                                    ->maxSize(5120) // 5MB
                                    ->helperText('صورة معاينة للموديل (اختياري)')
                                    ->columnSpan(1),
                            ]),
                    ]),

                Section::make('الحالة والإعدادات')
                    ->description('إعدادات العرض والحالة')
                    ->icon(Heroicon::OutlinedCog)
                    ->schema([
                        Grid::make(3)
                            ->schema([
                                Checkbox::make('is_active')
                                    ->label('نشط')
                                    ->helperText('عرض الموديل في المتجر')
                                    ->default(true)
                                    ->inline(false)
                                    ->columnSpan(1),

                                Checkbox::make('is_featured')
                                    ->label('مميز')
                                    ->helperText('عرض كموديل مميز')
                                    ->default(false)
                                    ->inline(false)
                                    ->columnSpan(1),

                                TextInput::make('order')
                                    ->label('الترتيب')
                                    ->numeric()
                                    ->default(0)
                                    ->helperText('ترتيب العرض (الأقل أولاً)')
                                    ->columnSpan(1),
                            ]),
                    ]),

                Section::make('إحصائيات')
                    ->description('المشاهدات والتصاميم')
                    ->icon(Heroicon::OutlinedChartBar)
                    ->schema([
                        Grid::make(2)
                            ->schema([
                                TextInput::make('views_count')
                                    ->label('المشاهدات')
                                    ->numeric()
                                    ->default(0)
                                    ->disabled()
                                    ->dehydrated(false)
                                    ->columnSpan(1),

                                TextInput::make('designs_count')
                                    ->label('عدد التصاميم')
                                    ->numeric()
                                    ->default(0)
                                    ->disabled()
                                    ->dehydrated(false)
                                    ->columnSpan(1),
                            ]),
                    ])
                    ->collapsible()
                    ->collapsed(),
            ]);
    }
}
