<?php

namespace App\Filament\Resources\ProductCards\Schemas;

use App\Models\Product;
use App\Models\ProductPrice;
use App\Models\ProductVariant;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Components\Utilities\Set;
use Filament\Schemas\Schema;

class ProductCardForm
{
    private static function moneyToInt(mixed $value): int
    {
        if ($value === null || $value === '') {
            return 0;
        }

        $raw = preg_replace('/[^\d]/', '', (string) $value);

        return (int) ($raw === '' ? 0 : $raw);
    }

    private static function formatInt(mixed $value): string
    {
        $v = self::moneyToInt($value);

        return $v > 0 ? number_format($v) : '';
    }

    /**
     * جلب سعر التجزئة للمنتج/المتغير من ProductPrice (أولوية) ثم fallback
     */
    private static function resolveRetailPrice(int $productId, ?int $variantId = null): int
    {
        if ($productId <= 0) {
            return 0;
        }

        // 1) من جدول product_prices (الأضمن)
        $q = ProductPrice::query()
            ->where('product_id', $productId)
            ->active();

        if ($variantId && $variantId > 0) {
            $q->where('product_variant_id', $variantId);
        } else {
            $q->whereNull('product_variant_id');
        }

        $price = (int) ($q->value('retail_price') ?? 0);
        if ($price > 0) {
            return $price;
        }

        // 2) fallback احتياطي من المنتج/المتغير إن كان عندك منطق تسعير آخر
        if ($variantId && $variantId > 0) {
            $variant = ProductVariant::query()->find($variantId);

            return (int) ($variant?->getPriceForType('retail') ?? 0);
        }

        // 3) إن لم يوجد variant، جرّب المتغير الافتراضي ثم fallback للمنتج
        $defaultVariantId = (int) (ProductVariant::query()
            ->where('product_id', $productId)
            ->where('is_default', true)
            ->value('id') ?? 0);

        if ($defaultVariantId > 0) {
            $price = (int) (ProductPrice::query()
                ->where('product_id', $productId)
                ->where('product_variant_id', $defaultVariantId)
                ->active()
                ->value('retail_price') ?? 0);

            if ($price > 0) {
                return $price;
            }

            $variant = ProductVariant::query()->find($defaultVariantId);
            $fallback = (int) ($variant?->getPriceForType('retail') ?? 0);
            if ($fallback > 0) {
                return $fallback;
            }
        }

        return 0;
    }

    public static function configure(Schema $schema): Schema
    {
        return $schema->components([
            Hidden::make('branch_id')
                ->default(fn () => (int) user_info('branch_id'))
                ->dehydrated(true),

            Hidden::make('user_id')
                ->default(fn () => (int) user_info('id'))
                ->dehydrated(true),

            Section::make('معلومات المنتج')
                ->description('اختر المنتج وحدد المتغير (اختياري) وسيتم جلب سعر المفرد تلقائياً')
                ->icon('heroicon-o-cube')
                ->iconColor('primary')
                ->collapsible()
                ->columnSpanFull()
                ->schema([
                    Grid::make(2)->schema([
                        Select::make('product_id')
                            ->label('المنتج')
                            ->placeholder('ابحث عن المنتج...')
                            ->required()
                            ->searchable()
                            ->preload()
                            ->native(false)
                            ->options(
                                fn () => Product::query()
                                    ->orderBy('name')
                                    ->pluck('name', 'id')
                                    ->toArray()
                            )
                            ->live(onBlur: true)
                            ->afterStateUpdated(function ($state, Set $set) {
                                $productId = (int) $state;

                                // عند تغيير المنتج: صفّر المتغير
                                $set('product_variant_id', null);

                                $productName = Product::query()
                                    ->whereKey($productId)
                                    ->value('name');

                                if ($productName) {
                                    $set('title', (string) $productName);
                                }

                                // ✅ جلب سعر المنتج (يفضل default variant إن كان موجوداً وإلا سعر المنتج)
                                $price = self::resolveRetailPrice($productId);

                                $set('price', $price > 0 ? number_format($price) : '');
                            })
                            ->prefixIcon('heroicon-o-shopping-bag'),

                        Select::make('product_variant_id')
                            ->label('المتغير')
                            ->placeholder('اختر المتغير (اختياري)...')
                            ->nullable()
                            ->searchable()
                            ->preload()
                            ->native(false)
                            ->options(function (Get $get) {
                                $productId = (int) ($get('product_id') ?? 0);
                                if ($productId <= 0) {
                                    return [];
                                }

                                return ProductVariant::query()
                                    ->where('product_id', $productId)
                                    ->orderBy('id', 'desc')
                                    ->get(['id'])
                                    ->mapWithKeys(function (ProductVariant $v) {
                                        $label = trim((string) ($v->full_name ?? ''));
                                        if ($label === '') {
                                            $label = 'متغير #'.(int) $v->id;
                                        }

                                        return [(int) $v->id => $label];
                                    })
                                    ->toArray();
                            })
                            ->disabled(fn (Get $get) => ! (int) ($get('product_id') ?? 0))
                            ->live(onBlur: true)
                            ->afterStateUpdated(function ($state, Get $get, Set $set) {
                                $productId = (int) ($get('product_id') ?? 0);
                                $variantId = (int) ($state ?? 0);

                                $productName = Product::query()
                                    ->whereKey($productId)
                                    ->value('name');

                                if (! $productName) {
                                    return;
                                }

                                if ($variantId > 0) {
                                    $variant = ProductVariant::query()->find($variantId);

                                    $variantLabel = (string) ($variant?->variant_label ?? '');
                                    $set(
                                        'title',
                                        $variantLabel !== ''
                                            ? trim($productName.' - '.$variantLabel)
                                            : (string) $productName
                                    );

                                    // ✅ جلب سعر المتغير
                                    $price = self::resolveRetailPrice($productId, $variantId);

                                    $set('price', $price > 0 ? number_format($price) : '');

                                    return;
                                }

                                // إذا أزال المتغير: رجّع عنوان المنتج وسعره
                                $set('title', (string) $productName);

                                $price = self::resolveRetailPrice($productId);

                                $set('price', $price > 0 ? number_format($price) : '');
                            })
                            ->prefixIcon('heroicon-o-tag'),
                    ]),
                ]),

            Section::make('بيانات العرض')
                ->description('كيف سيظهر المنتج على كرت السعر')
                ->icon('heroicon-o-identification')
                ->iconColor('success')
                ->collapsible()
                ->columnSpanFull()
                ->schema([
                    TextInput::make('title')
                        ->label('اسم المنتج على الكرت')
                        ->placeholder('سيتم ملؤه تلقائياً...')
                        ->required()
                        ->maxLength(120)
                        ->prefixIcon('heroicon-o-pencil-square')
                        ->columnSpanFull(),

                    Textarea::make('description')
                        ->label('الوصف')
                        ->placeholder('اكتب وصفاً مختصراً يظهر تحت اسم المنتج...')
                        ->rows(3)
                        ->maxLength(500)
                        ->columnSpanFull(),
                ]),

            // تم حذف قسم "صورة المنتج" بالكامل لأن الكارت يعتمد صورة المنتج نفسها

            Section::make('التسعير والخصم')
                ->description('سيتم جلب سعر المفرد تلقائياً ويمكنك تعديله')
                ->icon('heroicon-o-banknotes')
                ->iconColor('warning')
                ->collapsible()
                ->columnSpanFull()
                ->schema([
                    Grid::make(2)->schema([
                        TextInput::make('price')
                            ->label('السعر (مفرد)')
                            ->placeholder('أدخل السعر بالدينار العراقي')
                            ->required()
                            ->suffix('د.ع')
                            ->formatStateUsing(fn ($state) => self::formatInt($state))
                            ->dehydrateStateUsing(fn ($state) => self::moneyToInt($state))
                            ->extraInputAttributes([
                                'inputmode' => 'numeric',
                                'dir' => 'rtl',
                                'x-data' => '{}',
                                'x-on:input' => <<<'JS'
                                    let raw = ($el.value || '').replace(/[^\d]/g, '');
                                    $el.value = raw ? new Intl.NumberFormat('en-US').format(raw) : '';
                                JS,
                            ])
                            ->prefixIcon('heroicon-o-currency-dollar'),

                        TextInput::make('discount_percent')
                            ->label('نسبة الخصم')
                            ->placeholder('يمكنك تركه فارغاً إذا لم يكن هناك خصم')
                            ->suffix('%')
                            ->numeric()
                            ->minValue(1)
                            ->maxValue(99)
                            ->nullable()
                            ->prefixIcon('heroicon-o-receipt-percent'),

                        Toggle::make('show_discount')
                            ->label('عرض الخصم على الكرت')
                            ->inline(false)
                            ->default(false)
                            ->columnSpanFull(),
                    ]),
                ]),

            Section::make('الحالة')
                ->description('تفعيل أو إيقاف الكرت')
                ->icon('heroicon-o-cog-6-tooth')
                ->iconColor('gray')
                ->collapsible()
                ->collapsed()
                ->columnSpanFull()
                ->visibleOn('edit')
                ->schema([
                    Toggle::make('is_active')
                        ->label('نشط')
                        ->inline(false)
                        ->default(true),
                ]),

            Section::make('ملاحظات')
                ->description('ملاحظات داخلية في نهاية النموذج')
                ->icon('heroicon-o-chat-bubble-left-right')
                ->iconColor('secondary')
                ->collapsible()
                ->collapsed()
                ->columnSpanFull()
                ->schema([
                    Textarea::make('notes')
                        ->label('الملاحظات')
                        ->placeholder('اكتب أي ملاحظات داخلية هنا...')
                        ->rows(4)
                        ->maxLength(1000)
                        ->columnSpanFull(),
                ]),
        ]);
    }
}
