<?php

namespace App\Filament\Resources\Pos\Tables;

use App\Models\Invoice;
use App\Services\Invoices\DeleteInvoiceService;
use Filament\Actions\Action;
use Filament\Actions\ActionGroup;
use Filament\Actions\BulkAction;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Actions\RestoreAction;
use Filament\Actions\RestoreBulkAction;
use Filament\Actions\ViewAction;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Select;
use Filament\Notifications\Notification;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TrashedFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class PosTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->defaultSort('invoice_date', 'desc')
            ->columns([
                TextColumn::make('invoice_number')
                    ->label('رقم الفاتورة')
                    ->searchable()
                    ->sortable(),

                TextColumn::make('branch.name')
                    ->label('الفرع')
                    ->searchable()
                    ->visible(user_info()->admin())
                    ->toggleable(),

                TextColumn::make('client.name')
                    ->label('العميل')
                    ->searchable()
                    ->toggleable(),

                TextColumn::make('sale_type')
                    ->label('نوع البيع')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->colors([
                        'primary' => 'retail',
                        'success' => 'wholesale',
                        'warning' => 'agent',
                    ])
                    ->icons([
                        'heroicon-o-user' => 'retail',
                        'heroicon-o-users' => 'wholesale',
                        'heroicon-o-briefcase' => 'agent',
                    ])
                    ->formatStateUsing(fn ($state) => match ($state) {
                        'wholesale' => 'جملة',
                        'agent' => 'وكيل',
                        default => 'مفرد',
                    }),

                TextColumn::make('grand_total')
                    ->label('الإجمالي النهائي')
                    ->sortable()
                    ->weight('bold')
                    ->color('info')
                    ->formatStateUsing(fn ($state) => number_format((int) $state))
                    ->suffix(' دينار '),

                TextColumn::make('paid_amount')
                    ->label('المبلغ المدفوع')
                    ->sortable()
                    ->weight('bold')
                    ->color('success')
                    ->formatStateUsing(fn ($state) => number_format((int) $state))
                    ->suffix(' دينار '),

                TextColumn::make('due_amount')
                    ->label('المتبقي')
                    ->sortable()
                    ->weight('bold')
                    ->color('danger')
                    ->formatStateUsing(fn ($state) => number_format((int) $state))
                    ->suffix(' دينار '),

                TextColumn::make('user.name')
                    ->label('الموظف')
                    ->toggleable(),

                TextColumn::make('status')
                    ->label('حالة الفاتورة')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->colors([
                        'warning' => 'preparing',
                        'success' => 'completed',
                        'danger' => 'cancelled',
                    ])
                    ->icons([
                        'heroicon-o-clock' => 'preparing',
                        'heroicon-o-check-badge' => 'completed',
                        'heroicon-o-x-circle' => 'cancelled',
                    ])
                    ->formatStateUsing(fn ($state) => match ($state) {
                        'preparing' => 'قيد التجهيز',
                        'completed' => 'مكتملة',
                        'cancelled' => 'ملغاة',
                        default => '-',
                    }),

                TextColumn::make('payment_status')
                    ->label('حالة الدفع')
                    ->sortable()
                    ->colors([
                        'danger' => 'unpaid',
                        'warning' => 'partial',
                        'success' => 'paid',
                        'gray' => 'refunded',
                    ])
                    ->icons([
                        'heroicon-o-exclamation-circle' => 'unpaid',
                        'heroicon-o-adjustments-horizontal' => 'partial',
                        'heroicon-o-check-circle' => 'paid',
                        'heroicon-o-arrow-uturn-left' => 'refunded',
                    ])
                    ->formatStateUsing(fn ($state) => match ($state) {
                        'unpaid' => 'غير مدفوعة',
                        'partial' => 'مدفوعة جزئياً',
                        'paid' => 'مدفوعة',
                        'refunded' => 'مستردة',
                        default => '-',
                    })
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('invoice_date')
                    ->label('تاريخ الفاتورة')
                    ->date('Y-m-d')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->dateTime('Y-m-d H:i')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('updated_at')
                    ->label('آخر تعديل')
                    ->dateTime('Y-m-d H:i')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('deleted_at')
                    ->label('تاريخ الحذف')
                    ->dateTime('Y-m-d H:i')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                TrashedFilter::make(),

                SelectFilter::make('status')
                    ->label('حالة الفاتورة')
                    ->options([
                        'preparing' => 'قيد التجهيز',
                        'completed' => 'مكتملة',
                        'cancelled' => 'ملغاة',
                    ])
                    ->searchable()
                    ->preload(),

                SelectFilter::make('payment_status')
                    ->label('حالة الدفع')
                    ->options([
                        'unpaid' => 'غير مدفوعة',
                        'partial' => 'مدفوعة جزئياً',
                        'paid' => 'مدفوعة بالكامل',
                        'refunded' => 'مستردة',
                    ])
                    ->searchable()
                    ->preload(),

                Filter::make('date_range')
                    ->label('الفترة الزمنية')
                    ->schema([
                        Select::make('period')
                            ->label('اختر الفترة')
                            ->options([
                                'today' => 'اليوم',
                                'yesterday' => 'الأمس',
                                'this_week' => 'هذا الأسبوع',
                                'last_week' => 'الأسبوع الماضي',
                                'this_month' => 'هذا الشهر',
                                'last_month' => 'الشهر الماضي',
                                'custom' => 'مخصص',
                            ])
                            ->reactive()
                            ->afterStateUpdated(
                                fn ($state, $set) => $state !== 'custom'
                                    ? ($set('date_from', null) || $set('date_to', null))
                                    : null
                            ),

                        DatePicker::make('date_from')
                            ->label('من تاريخ')
                            ->native(false)
                            ->visible(fn ($get) => $get('period') === 'custom'),

                        DatePicker::make('date_to')
                            ->label('إلى تاريخ')
                            ->native(false)
                            ->visible(fn ($get) => $get('period') === 'custom'),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        $period = $data['period'] ?? null;
                        if (! $period) {
                            return $query;
                        }

                        return match ($period) {
                            'today' => $query->whereDate('invoice_date', today()),
                            'yesterday' => $query->whereDate('invoice_date', today()->subDay()),
                            'this_week' => $query->whereBetween('invoice_date', [now()->startOfWeek(), now()->endOfWeek()]),
                            'last_week' => $query->whereBetween('invoice_date', [now()->subWeek()->startOfWeek(), now()->subWeek()->endOfWeek()]),
                            'this_month' => $query->whereMonth('invoice_date', now()->month)->whereYear('invoice_date', now()->year),
                            'last_month' => $query->whereMonth('invoice_date', now()->subMonth()->month)->whereYear('invoice_date', now()->subMonth()->year),
                            'custom' => $query
                                ->when($data['date_from'] ?? null, fn ($q, $v) => $q->whereDate('invoice_date', '>=', $v))
                                ->when($data['date_to'] ?? null, fn ($q, $v) => $q->whereDate('invoice_date', '<=', $v)),
                            default => $query,
                        };
                    }),
            ])
            ->recordActions([
                ActionGroup::make([
                    ViewAction::make()->label('عرض'),
                    EditAction::make()->label('تعديل'),

                    RestoreAction::make()
                        ->label('استرجاع')
                        ->visible(fn ($record) => filled($record?->deleted_at)),

                    // ✅ بدل ForceDeleteAction الافتراضي: Action مخصص يستدعي الخدمة
                    Action::make('force_delete_with_effects')
                        ->label('حذف نهائي')
                        ->color('danger')
                        ->icon('heroicon-o-trash')
                        ->requiresConfirmation()
                        ->modalHeading('حذف نهائي للفاتورة')
                        ->modalDescription('سيتم سحب المبالغ من الخزنة، وإرجاع المخزون، وحذف الأقساط والوصولات وكل ما يتعلق بهذه الفاتورة بشكل نهائي.')
                        ->visible(fn ($record) => filled($record?->deleted_at))
                        ->action(function (Invoice $record) {
                            app(DeleteInvoiceService::class)->executeForce($record);

                            Notification::make()
                                ->title('تم الحذف النهائي بنجاح')
                                ->body('تم سحب المبلغ من الخزنة وإرجاع المخزون وحذف الأقساط والوصولات.')
                                ->success()
                                ->send();
                        }),
                ]),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make()->label('حذف'),
                    RestoreBulkAction::make()->label('استرجاع'),

                    // ✅ بدل ForceDeleteBulkAction الافتراضي: BulkAction مخصص
                    BulkAction::make('force_delete_bulk_with_effects')
                        ->label('حذف نهائي')
                        ->color('danger')
                        ->icon('heroicon-o-trash')
                        ->requiresConfirmation()
                        ->modalHeading('حذف نهائي للفاتورة/الفواتير')
                        ->modalDescription('سيتم سحب المبالغ من الخزنة وإرجاع المخزون وحذف الأقساط والوصولات لكل فاتورة محددة.')
                        ->action(function ($records) {
                            $service = app(DeleteInvoiceService::class);

                            foreach ($records as $record) {
                                /** @var Invoice $record */
                                $service->executeForce($record);
                            }

                            Notification::make()
                                ->title('تم الحذف النهائي للفواتير المحددة')
                                ->success()
                                ->send();
                        }),
                ]),
            ]);
    }
}
