<?php

namespace App\Filament\Resources\Pos\PosForm;

use App\Models\BranchSetting;
use App\Models\Client;
use App\Models\InvoiceCounter;
use App\Models\Product;
use App\Models\ProductVariant;
use App\Models\StockBalance;
use App\Models\Warehouse;
use Coolsam\Flatpickr\Forms\Components\Flatpickr;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Notifications\Notification;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Components\Utilities\Set;
use Filament\Schemas\Schema;
use Filament\Support\RawJs;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;

class PosForm
{
    /** ✅ حسب طلبك */
    private const NUM_ATTRS = ['dir' => 'ltr', 'style' => 'text-align:right'];

    /** لمنع تكرار إشعار "المحصّل فعلياً" */
    private static bool $warnedMinPaid = false;

    /* ============================ Helpers: أرقام صحيحة + تنسيق ============================ */

    private static function toInt(mixed $v): int
    {
        if ($v === null || $v === '') {
            return 0;
        }
        if (is_int($v)) {
            return $v;
        }
        if (is_float($v)) {
            return (int) $v;
        }

        $s = (string) $v;

        $s = strtr($s, [
            '٠' => '0',
            '١' => '1',
            '٢' => '2',
            '٣' => '3',
            '٤' => '4',
            '٥' => '5',
            '٦' => '6',
            '٧' => '7',
            '٨' => '8',
            '٩' => '9',
            '۰' => '0',
            '۱' => '1',
            '۲' => '2',
            '۳' => '3',
            '۴' => '4',
            '۵' => '5',
            '۶' => '6',
            '۷' => '7',
            '۸' => '8',
            '۹' => '9',
        ]);

        $s = preg_replace('/[^\d]/', '', $s);

        return (int) ($s === '' ? 0 : $s);
    }

    private static function formatInt(mixed $v): string
    {
        return number_format(self::toInt($v), 0, '', ',');
    }

    /* ============================ Mask: يرسل digits فقط للسيرفر ============================ */

    private static function digitsOnlyMask(): RawJs
    {
        return RawJs::make(<<<'JS'
(() => {
    const map = {
        '٠':'0','١':'1','٢':'2','٣':'3','٤':'4','٥':'5','٦':'6','٧':'7','٨':'8','٩':'9',
        '۰':'0','۱':'1','۲':'2','۳':'3','۴':'4','۵':'5','۶':'6','۷':'7','۸':'8','۹':'9',
    };
    let s = ($input ?? '').toString();
    s = s.split('').map(c => map[c] ?? c).join('');
    s = s.replace(/[^0-9]/g, '');
    return s;
})()
JS);
    }

    /**
     * ✅ حقل أموال/أعداد صحيحة:
     * - التنسيق بالفواصل عند انتهاء الكتابة فقط (onBlur)
     * - حفظ int
     */
    private static function moneyInput(string $name): TextInput
    {
        return TextInput::make($name)
            ->mask(self::digitsOnlyMask())
            ->stripCharacters([',', ' '])
            ->inputMode('numeric')
            ->rule('integer')
            ->live(onBlur: true)
            ->afterStateUpdated(function (Set $set, $state) use ($name) {
                $set($name, self::formatInt($state));
            })
            ->formatStateUsing(fn ($state) => self::formatInt($state))
            ->dehydrateStateUsing(fn ($state) => self::toInt($state))
            ->extraInputAttributes(self::NUM_ATTRS);
    }

    /* ============================ Root Get/Set Helpers ============================ */

    private static function getRootArray(Get $get, string $field): array
    {
        $v = $get($field);
        if (is_array($v)) {
            return $v;
        }

        $v = $get('../'.$field);
        if (is_array($v)) {
            return $v;
        }

        $v = $get('../../'.$field);
        if (is_array($v)) {
            return $v;
        }

        return [];
    }

    private static function getRootScalar(Get $get, string $field, $default = null)
    {
        $v = $get($field);
        if ($v !== null) {
            return $v;
        }

        $v = $get('../'.$field);
        if ($v !== null) {
            return $v;
        }

        $v = $get('../../'.$field);
        if ($v !== null) {
            return $v;
        }

        return $default;
    }

    private static function setRoot(Set $set, Get $get, string $field, $value): void
    {
        if ($get($field) !== null) {
            $set($field, $value);

            return;
        }

        if ($get('../'.$field) !== null) {
            $set('../'.$field, $value);

            return;
        }

        $set('../../'.$field, $value);
    }

    /* ============================ Settings ============================ */

    private static function posSettings(): array
    {
        static $cache = null;
        if ($cache !== null) {
            return $cache;
        }

        $branchId = (int) user_info('branch_id');

        $fallback = [
            'defaults' => ['sale_type' => 'retail'],
            'features' => ['show_notes' => true],
        ];

        if (! $branchId) {
            return $cache = $fallback;
        }

        $record = BranchSetting::query()
            ->where('branch_id', $branchId)
            ->first(['id', 'branch_id', 'invoice_settings']);

        $raw = is_array($record?->invoice_settings) ? $record->invoice_settings : [];

        $defaults = [
            'sale_type' => (string) data_get($raw, 'defaults.sale_type', $fallback['defaults']['sale_type']),
        ];

        $features = [
            'show_notes' => (bool) data_get($raw, 'features.show_notes', $fallback['features']['show_notes']),
        ];

        if (! in_array($defaults['sale_type'], ['retail', 'wholesale', 'agent'], true)) {
            $defaults['sale_type'] = 'retail';
        }

        return $cache = compact('defaults', 'features');
    }

    private static function invoiceDefault(string $key, $fallback = null)
    {
        return data_get(self::posSettings(), "defaults.$key", $fallback);
    }

    private static function invoiceFeature(string $key, bool $fallback = true): bool
    {
        return (bool) data_get(self::posSettings(), "features.$key", $fallback);
    }

    private static function getSaleType(Get $get): string
    {
        $saleType = (string) (self::getRootScalar(
            $get,
            'sale_type',
            (string) self::invoiceDefault('sale_type', 'retail')
        ) ?? 'retail');

        return in_array($saleType, ['retail', 'wholesale', 'agent'], true) ? $saleType : 'retail';
    }

    /* ============================ Warehouses + Stock (كاش) ============================ */

    private static function currentBranchWarehouseIds(): array
    {
        static $cache = null;
        if ($cache !== null) {
            return $cache;
        }

        $branchId = (int) user_info('branch_id');
        if ($branchId <= 0) {
            return $cache = [];
        }

        return $cache = Warehouse::query()
            ->forBranch($branchId)
            ->active()
            ->pluck('warehouses.id')
            ->map(fn ($id) => (int) $id)
            ->values()
            ->all();
    }

    private static function variantAvailableQtyInBranch(int $variantId, array $warehouseIds): int
    {
        if ($variantId <= 0 || empty($warehouseIds)) {
            return 0;
        }

        $sum = (int) StockBalance::query()
            ->whereIn('warehouse_id', $warehouseIds)
            ->where('product_variant_id', $variantId)
            ->selectRaw('COALESCE(SUM(on_hand - reserved_qty), 0) as available_sum')
            ->value('available_sum');

        return max(0, $sum);
    }

    private static function variantIsAvailableInBranch(int $variantId, array $warehouseIds): bool
    {
        if ($variantId <= 0 || empty($warehouseIds)) {
            return false;
        }

        return StockBalance::query()
            ->whereIn('warehouse_id', $warehouseIds)
            ->where('product_variant_id', $variantId)
            ->whereRaw('(on_hand - reserved_qty) > 0')
            ->exists();
    }

    /* ============================ Product Label Helpers ============================ */

    private static function cleanDesc(?string $desc, int $max = 50): string
    {
        $desc = trim((string) $desc);
        if ($desc === '') {
            return '';
        }

        $desc = preg_replace("/\s+/u", ' ', $desc);
        if (mb_strlen($desc) > $max) {
            $desc = mb_substr($desc, 0, $max).'…';
        }

        return $desc;
    }

    private static function productLabel(string $name, ?string $desc): string
    {
        $name = trim($name);
        $d = self::cleanDesc($desc);

        return $d !== '' ? ($name.' — '.$d) : $name;
    }

    /* ============================ Products Search (فصل المنتج) ============================ */

    private static function searchAvailableProducts(string $search, array $warehouseIds, int $limit = 50): array
    {
        if (empty($warehouseIds)) {
            return [];
        }

        $search = trim($search);

        $rows = Product::query()
            ->when($search !== '', function ($q) use ($search) {
                $q->where(function ($qq) use ($search) {
                    $qq->where('name', 'like', "%{$search}%")
                        ->orWhere('description', 'like', "%{$search}%");
                });
            })
            ->whereExists(function ($q) use ($warehouseIds) {
                $q->selectRaw('1')
                    ->from('product_variants as pv')
                    ->whereColumn('pv.product_id', 'products.id')
                    ->whereNull('pv.deleted_at')
                    ->where('pv.is_active', 1)
                    ->whereExists(function ($qq) use ($warehouseIds) {
                        $qq->selectRaw('1')
                            ->from('stock_balances as sb')
                            ->whereColumn('sb.product_variant_id', 'pv.id')
                            ->whereIn('sb.warehouse_id', $warehouseIds)
                            ->whereRaw('(sb.on_hand - sb.reserved_qty) > 0');
                    });
            })
            ->orderBy('name')
            ->limit($limit)
            ->get(['id', 'name', 'description']);

        if ($rows->isEmpty()) {
            return [];
        }

        return $rows->mapWithKeys(function ($p) {
            $id = (int) $p->id;

            return [$id => self::productLabel((string) $p->name, (string) ($p->description ?? ''))];
        })->all();
    }

    private static function getProductLabelById(int $productId): string
    {
        static $cache = [];
        if ($productId <= 0) {
            return '—';
        }
        if (isset($cache[$productId])) {
            return $cache[$productId];
        }

        $row = Product::query()
            ->whereKey($productId)
            ->first(['id', 'name', 'description']);

        if (! $row) {
            return $cache[$productId] = '—';
        }

        return $cache[$productId] = self::productLabel((string) $row->name, (string) ($row->description ?? ''));
    }

    /* ============================ Variant Options (فصل المتغير) ============================ */

    private static function getVariantOptionsOnlyFast(int $productId, array $warehouseIds): array
    {
        static $cache = [];
        $key = $productId.':'.md5(json_encode($warehouseIds));
        if (isset($cache[$key])) {
            return $cache[$key];
        }

        if ($productId <= 0 || empty($warehouseIds)) {
            return $cache[$key] = [];
        }

        $variants = ProductVariant::query()
            ->where('product_variants.product_id', $productId)
            ->where('product_variants.is_active', 1)
            ->whereNull('product_variants.deleted_at')
            ->whereExists(function ($q) use ($warehouseIds) {
                $q->selectRaw('1')
                    ->from('stock_balances as sb')
                    ->whereColumn('sb.product_variant_id', 'product_variants.id')
                    ->whereIn('sb.warehouse_id', $warehouseIds)
                    ->whereRaw('(sb.on_hand - sb.reserved_qty) > 0');
            })
            ->orderBy('product_variants.id')
            ->get(['product_variants.id', 'product_variants.sku']);

        if ($variants->isEmpty()) {
            return $cache[$key] = [];
        }

        $ids = $variants->pluck('id')->map(fn ($x) => (int) $x)->values()->all();

        $rows = DB::table('variant_attribute_values as vav')
            ->join('attribute_values as av', 'av.id', '=', 'vav.attribute_value_id')
            ->join('attributes as a', 'a.id', '=', 'vav.attribute_id')
            ->whereIn('vav.variant_id', $ids)
            ->orderBy('a.sort_order')
            ->orderBy('a.id')
            ->orderBy('av.sort_order')
            ->orderBy('av.id')
            ->get([
                'vav.variant_id',
                'av.value as value_raw',
                'av.display_value as value_display',
            ]);

        $grouped = [];
        foreach ($rows as $r) {
            $vid = (int) $r->variant_id;
            $val = trim((string) ($r->value_display ?: $r->value_raw ?: ''));
            if ($val === '') {
                continue;
            }
            $grouped[$vid][] = $val;
        }

        $out = [];
        foreach ($variants as $v) {
            $vid = (int) $v->id;
            $values = $grouped[$vid] ?? [];
            $out[$vid] = $values ? implode(' - ', $values) : ((string) ($v->sku ?: ('متغير #'.$vid)));
        }

        return $cache[$key] = $out;
    }

    private static function variantOnlyLabelById(int $variantId): string
    {
        static $cache = [];
        if ($variantId <= 0) {
            return '—';
        }
        if (isset($cache[$variantId])) {
            return $cache[$variantId];
        }

        $rows = DB::table('variant_attribute_values as vav')
            ->join('attribute_values as av', 'av.id', '=', 'vav.attribute_value_id')
            ->join('attributes as a', 'a.id', '=', 'vav.attribute_id')
            ->where('vav.variant_id', $variantId)
            ->orderBy('a.sort_order')
            ->orderBy('a.id')
            ->orderBy('av.sort_order')
            ->orderBy('av.id')
            ->get(['av.value as value_raw', 'av.display_value as value_display']);

        if ($rows->isEmpty()) {
            $sku = (string) (ProductVariant::query()->whereKey($variantId)->value('sku') ?? '');

            return $cache[$variantId] = ($sku !== '' ? $sku : ('متغير #'.$variantId));
        }

        $values = $rows->map(fn ($r) => trim((string) ($r->value_display ?: $r->value_raw ?: '')))
            ->filter(fn ($v) => $v !== '')
            ->values()
            ->all();

        return $cache[$variantId] = ($values ? implode(' - ', $values) : ('متغير #'.$variantId));
    }

    private static function itemLabelFromState(array $state): string
    {
        $productId = (int) ($state['product_id'] ?? 0);
        $variantId = (int) ($state['product_variant_id'] ?? 0);

        if ($productId <= 0 && $variantId <= 0) {
            return 'عنصر جديد';
        }

        $pLabel = self::getProductLabelById($productId);
        if ($variantId <= 0) {
            return $pLabel;
        }

        return $pLabel.' - '.self::variantOnlyLabelById($variantId);
    }

    /* ============================ Pricing (كاش) ============================ */

    private static function getVariantUnitPrice(int $variantId, string $saleType): int
    {
        static $cache = [];
        if ($variantId <= 0) {
            return 0;
        }

        $key = $variantId.':'.$saleType;
        if (isset($cache[$key])) {
            return $cache[$key];
        }

        $variant = ProductVariant::query()->find($variantId);
        $price = $variant ? max(0, (int) $variant->getPriceForType($saleType)) : 0;

        return $cache[$key] = $price;
    }

    /* ============================ Calculations ============================ */

    private static function calculateLineTotal(Set $set, Get $get): void
    {
        $price = self::toInt($get('unit_price'));
        $qty = max(1, self::toInt($get('quantity')));

        $total = ($price > 0 && $qty > 0) ? ($price * $qty) : 0;
        $set('line_total', self::formatInt($total));
    }

    private static function calculateItemsTotal(Get $get): int
    {
        $sum = 0;
        $items = self::getRootArray($get, 'ready_items');

        foreach ($items as $item) {
            $price = self::toInt($item['unit_price'] ?? 0);
            $qty = self::toInt($item['quantity'] ?? 0);
            $sum += ($price > 0 && $qty > 0) ? ($price * $qty) : 0;
        }

        return $sum;
    }

    private static function calculateNetTotal(Get $get): int
    {
        $itemsTotal = self::calculateItemsTotal($get);
        $shipping = self::toInt(self::getRootScalar($get, 'shipping', 0));
        $discount = self::toInt(self::getRootScalar($get, 'discount', 0));

        return max(0, $itemsTotal + $shipping - $discount);
    }

    private static function updateAllTotals(Set $set, Get $get): void
    {
        $itemsTotal = self::calculateItemsTotal($get);
        $discount = self::toInt(self::getRootScalar($get, 'discount', 0));
        $shipping = self::toInt(self::getRootScalar($get, 'shipping', 0));
        $netTotal = max(0, $itemsTotal + $shipping - $discount);

        $minPaid = self::toInt(self::getRootScalar($get, 'min_paid_amount', 0));

        $paidRaw = self::toInt(self::getRootScalar($get, 'paid_amount', 0));
        $paid = min($paidRaw, $netTotal);
        $paid = max($paid, $minPaid);

        if ($paid !== $paidRaw) {
            self::setRoot($set, $get, 'paid_amount', self::formatInt($paid));

            if ($minPaid > 0 && $paidRaw < $minPaid && ! self::$warnedMinPaid) {
                self::$warnedMinPaid = true;

                Notification::make()
                    ->title('تنبيه')
                    ->body('لا يمكن أن يكون المدفوع أقل من المبالغ المحصّلة فعلياً: '.self::formatInt($minPaid))
                    ->warning()
                    ->send();
            }
        } else {
            self::$warnedMinPaid = false;
        }

        $due = max(0, $netTotal - $paid);

        self::setRoot($set, $get, 'subtotal', self::formatInt($itemsTotal));
        self::setRoot($set, $get, 'net_total', self::formatInt($netTotal));
        self::setRoot($set, $get, 'total_due', self::formatInt($due));

        self::setRoot($set, $get, 'invoice_subtotal', self::formatInt($itemsTotal));
        self::setRoot($set, $get, 'invoice_discount', self::formatInt($discount));
        self::setRoot($set, $get, 'invoice_shipping', self::formatInt($shipping));
        self::setRoot($set, $get, 'invoice_paid', self::formatInt($paid));

        self::setRoot($set, $get, 'total_discount', self::formatInt($discount));
        self::setRoot($set, $get, 'total_shipping', self::formatInt($shipping));
        self::setRoot($set, $get, 'grand_total', self::formatInt($netTotal));
        self::setRoot($set, $get, 'total_paid', self::formatInt($paid));
        self::setRoot($set, $get, 'total_due', self::formatInt($due));
    }

    /* ============================ Business ============================ */

    /**
     * ✅ Preview فقط (ليس للحفظ)
     * هذا الرقم "متوقع" وقد لا يطابق الرقم النهائي عند الحفظ بسبب التزامن.
     */
    private static function peekInvoiceNumber(Get $get): string
    {
        $branchId = (int) user_info('branch_id');
        if (! $branchId) {
            return '000001';
        }

        $counter = InvoiceCounter::query()->where('branch_id', $branchId)->first();
        $next = $counter ? ((int) $counter->current_sequence + 1) : 1;

        return str_pad((string) $next, 6, '0', STR_PAD_LEFT);
    }

    private static function fetchPriceFromVariant(Set $set, Get $get): void
    {
        $variantId = (int) ($get('product_variant_id') ?? 0);

        $set('unit_price', self::formatInt(0));
        $set('line_total', self::formatInt(0));
        $set('available_quantity', self::formatInt(0));

        if ($variantId <= 0) {
            self::updateAllTotals($set, $get);

            return;
        }

        $warehouseIds = self::currentBranchWarehouseIds();

        if (! self::variantIsAvailableInBranch($variantId, $warehouseIds)) {
            self::updateAllTotals($set, $get);

            return;
        }

        $saleType = self::getSaleType($get);
        $unitPrice = self::getVariantUnitPrice($variantId, $saleType);
        $available = self::variantAvailableQtyInBranch($variantId, $warehouseIds);

        $set('available_quantity', self::formatInt($available));
        $set('unit_price', self::formatInt($unitPrice));

        $qty = max(1, self::toInt($get('quantity') ?: 1));
        $set('quantity', self::formatInt($qty));

        self::calculateLineTotal($set, $get);
        self::updateAllTotals($set, $get);
    }

    private static function repriceRepeaterItems(Set $set, Get $get): void
    {
        $rows = self::getRootArray($get, 'ready_items');
        if (! $rows) {
            self::updateAllTotals($set, $get);

            return;
        }

        $saleType = self::getSaleType($get);
        $warehouseIds = self::currentBranchWarehouseIds();

        foreach ($rows as $i => $row) {
            $variantId = (int) ($row['product_variant_id'] ?? 0);

            if ($variantId <= 0) {
                $set("ready_items.{$i}.unit_price", self::formatInt(0));
                $set("ready_items.{$i}.line_total", self::formatInt(0));
                $set("ready_items.{$i}.available_quantity", self::formatInt(0));

                continue;
            }

            $unitPrice = self::getVariantUnitPrice($variantId, $saleType);
            $available = self::variantAvailableQtyInBranch($variantId, $warehouseIds);
            $qty = max(1, self::toInt($row['quantity'] ?? 1));

            $set("ready_items.{$i}.available_quantity", self::formatInt($available));
            $set("ready_items.{$i}.unit_price", self::formatInt($unitPrice));
            $set("ready_items.{$i}.quantity", self::formatInt($qty));
            $set("ready_items.{$i}.line_total", self::formatInt($unitPrice * $qty));
        }

        self::updateAllTotals($set, $get);
    }

    /* ============================ Schema ============================ */

    public static function configure(Schema $schema): Schema
    {
        return $schema
            ->columns(1)
            ->components([

                Section::make('معلومات الفاتورة')
                    ->description('البيانات الأساسية للبيع المباشر (جاهز فقط)')
                    ->icon('heroicon-o-document-text')
                    ->iconColor('primary')
                    ->collapsible()
                    ->persistCollapsed()
                    ->compact()
                    ->schema([

                        Hidden::make('branch_id')
                            ->required()
                            ->dehydrated()
                            ->default(fn () => (int) user_info('branch_id')),

                        Hidden::make('user_id')
                            ->dehydrated()
                            ->default(fn () => (int) user_info('id')),

                        Hidden::make('invoice_type')
                            ->default('ready')
                            ->dehydrated(true),

                        Hidden::make('min_paid_amount')
                            ->default(0)
                            ->dehydrated(false),

                        Grid::make(3)->schema([

                            TextInput::make('invoice_number')
                                ->label('رقم الفاتورة')
                                ->disabled()
                                ->dehydrated(false) // ✅ مهم: Preview فقط — لا يُرسل للسيرفر
                                ->default(fn (Get $get) => self::peekInvoiceNumber($get))
                                ->maxLength(6)
                                ->minLength(6)
                                ->extraInputAttributes(self::NUM_ATTRS),

                            Select::make('client_id')
                                ->label('العميل')
                                ->relationship('client', 'name')
                                ->searchable()
                                ->preload()
                                ->required()
                                ->native(false)
                                ->createOptionForm([
                                    Grid::make(2)->schema([
                                        TextInput::make('name')->label('اسم العميل')->required()->maxLength(190)->columnSpan(2),
                                        TextInput::make('phone')
                                            ->label('الهاتف')
                                            ->required()
                                            ->tel()
                                            ->maxLength(14),
                                        TextInput::make('phone2')
                                            ->label('هاتف إضافي')
                                            ->tel()
                                            ->maxLength(14),
                                        TextInput::make('company')->label('الشركة / العنوان التجاري')->maxLength(190),
                                        Textarea::make('address')->label('العنوان')->rows(2)->columnSpan(2),
                                        Hidden::make('branch_id')->default(fn () => (int) user_info('branch_id'))->dehydrated(true),
                                        Hidden::make('is_active')->default(true)->dehydrated(true),
                                    ]),
                                ])
                                ->createOptionUsing(function (array $data) {
                                    $data['branch_id'] = (int) ($data['branch_id'] ?? user_info('branch_id') ?? 0);
                                    $data['is_active'] = (bool) ($data['is_active'] ?? true);

                                    return Client::query()->create($data)->getKey();
                                })
                                ->createOptionAction(fn ($action) => $action->label('إضافة عميل جديد')),

                            Select::make('sale_type')
                                ->label('نوع البيع')
                                ->options(['retail' => 'مفرد', 'wholesale' => 'جملة', 'agent' => 'وكيل'])
                                ->required()
                                ->default(fn () => (string) self::invoiceDefault('sale_type', 'retail'))
                                ->native(false)
                                ->live(debounce: 150)
                                ->afterStateUpdated(fn (Set $set, Get $get) => self::repriceRepeaterItems($set, $get)),

                            Flatpickr::make('invoice_date')
                                ->label('تاريخ الفاتورة')
                                ->required()
                                ->default(now('Asia/Baghdad')->toDateString())
                                ->displayFormat('Y-m-d')
                                ->native(false)
                                ->dehydrateStateUsing(function ($state) {
                                    if (! $state) {
                                        return null;
                                    }
                                    try {
                                        return Carbon::parse($state)->toDateString();
                                    } catch (\Throwable) {
                                        return now('Asia/Baghdad')->toDateString();
                                    }
                                }),

                            Select::make('payment_status')
                                ->label('حالة الدفع')
                                ->options(['paid' => 'نقدي', 'installment' => 'أقساط'])
                                ->required()
                                ->default('paid')
                                ->native(false)
                                ->live(debounce: 150)
                                ->afterStateUpdated(function ($state, Set $set, Get $get) {
                                    $minPaid = self::toInt(self::getRootScalar($get, 'min_paid_amount', 0));
                                    $net = self::calculateNetTotal($get);

                                    if ($state === 'paid') {
                                        $paid = max($net, $minPaid);
                                        $set('paid_amount', self::formatInt($paid));
                                    } else {
                                        $set('paid_amount', self::formatInt($minPaid));
                                    }

                                    self::updateAllTotals($set, $get);
                                }),

                            Textarea::make('notes')
                                ->label('ملاحظات')
                                ->rows(3)
                                ->columnSpan(3)
                                ->visible(fn () => self::invoiceFeature('show_notes', true))
                                ->dehydrated(fn () => self::invoiceFeature('show_notes', true)),
                        ]),
                    ]),

                Section::make('عناصر الفاتورة')
                    ->description('المنتج منفصل عن المتغير')
                    ->icon('heroicon-o-shopping-bag')
                    ->iconColor('success')
                    ->collapsible()
                    ->collapsed(false)
                    ->compact()
                    ->schema([
                        Repeater::make('ready_items')
                            ->label('عناصر الفاتورة')
                            ->hiddenLabel()
                            ->defaultItems(1)
                            ->reorderable(false)
                            ->cloneable()
                            ->collapsible()
                            ->itemLabel(fn (array $state): ?string => self::itemLabelFromState($state))
                            ->schema([
                                Grid::make(12)
                                    ->schema([
                                        Select::make('product_id')
                                            ->label('المنتج')
                                            ->native(false)
                                            ->searchable()
                                            ->preload()
                                            ->required()
                                            ->live(debounce: 350)
                                            ->options(function () {
                                                $warehouseIds = self::currentBranchWarehouseIds();

                                                return self::searchAvailableProducts('', $warehouseIds, 60);
                                            })
                                            ->getSearchResultsUsing(function (string $search) {
                                                $warehouseIds = self::currentBranchWarehouseIds();

                                                return self::searchAvailableProducts($search, $warehouseIds, 60);
                                            })
                                            ->getOptionLabelUsing(fn ($value) => self::getProductLabelById((int) $value))
                                            ->afterStateUpdated(function (Set $set, Get $get) {
                                                $set('product_variant_id', null);
                                                $set('unit_price', self::formatInt(0));
                                                $set('available_quantity', self::formatInt(0));
                                                $set('quantity', self::formatInt(1));
                                                $set('line_total', self::formatInt(0));
                                                self::updateAllTotals($set, $get);
                                            })
                                            ->columnSpan(4),

                                        Select::make('product_variant_id')
                                            ->label('المتغير')
                                            ->native(false)
                                            ->searchable()
                                            ->preload()
                                            ->required()
                                            ->disabled(fn (Get $get) => ! $get('product_id'))
                                            ->options(function (Get $get) {
                                                $productId = (int) ($get('product_id') ?? 0);
                                                $warehouseIds = self::currentBranchWarehouseIds();

                                                return self::getVariantOptionsOnlyFast($productId, $warehouseIds);
                                            })
                                            ->live(debounce: 350)
                                            ->afterStateUpdated(function ($state, Set $set, Get $get) {
                                                if ($state) {
                                                    if (! $get('quantity')) {
                                                        $set('quantity', self::formatInt(1));
                                                    }
                                                    self::fetchPriceFromVariant($set, $get);
                                                } else {
                                                    $set('unit_price', self::formatInt(0));
                                                    $set('available_quantity', self::formatInt(0));
                                                    $set('line_total', self::formatInt(0));
                                                    self::updateAllTotals($set, $get);
                                                }
                                            })
                                            ->columnSpan(4),

                                        TextInput::make('available_quantity')
                                            ->label('المتوفر')
                                            ->suffix('قطعة')
                                            ->disabled()
                                            ->dehydrated(false)
                                            ->formatStateUsing(fn ($s) => self::formatInt($s))
                                            ->extraInputAttributes(self::NUM_ATTRS)
                                            ->columnSpan(4),

                                        self::moneyInput('quantity')
                                            ->label('الكمية')
                                            ->required()
                                            ->default(1)
                                            ->minValue(1)
                                            ->suffix('قطعة')
                                            ->afterStateUpdated(function (Set $set, Get $get, $state) {
                                                self::calculateLineTotal($set, $get);
                                                self::updateAllTotals($set, $get);

                                                $requested = self::toInt($state);
                                                $available = self::toInt($get('available_quantity'));

                                                if ($available > 0 && $requested > $available) {
                                                    Notification::make()
                                                        ->title('الكمية المطلوبة تتجاوز المتوفر')
                                                        ->body('المطلوب: '.self::formatInt($requested).' | المتوفر: '.self::formatInt($available))
                                                        ->warning()
                                                        ->send();
                                                }
                                            })
                                            ->columnSpan(4),

                                        self::moneyInput('unit_price')
                                            ->label('سعر القطعة')
                                            ->required()
                                            ->minValue(1)
                                            ->suffix('د.ع')
                                            ->afterStateUpdated(function (Set $set, Get $get) {
                                                self::calculateLineTotal($set, $get);
                                                self::updateAllTotals($set, $get);
                                            })
                                            ->columnSpan(4),

                                        TextInput::make('line_total')
                                            ->label('المجموع')
                                            ->suffix('د.ع')
                                            ->disabled()
                                            ->dehydrated(true)
                                            ->formatStateUsing(fn ($s) => self::formatInt($s))
                                            ->dehydrateStateUsing(fn ($s) => self::toInt($s))
                                            ->extraInputAttributes(self::NUM_ATTRS)
                                            ->columnSpan(4),

                                        Textarea::make('notes')
                                            ->label('ملاحظات')
                                            ->rows(1)
                                            ->columnSpan(12)
                                            ->visible(fn () => self::invoiceFeature('show_notes', true))
                                            ->dehydrated(fn () => self::invoiceFeature('show_notes', true)),

                                        Hidden::make('measurement')->default(1)->dehydrated(true),
                                        Hidden::make('item_type')->default('product')->dehydrated(true),
                                    ]),
                            ])
                            ->afterStateUpdated(fn (Set $set, Get $get) => self::updateAllTotals($set, $get)),
                    ]),

                Section::make('المبالغ المالية')
                    ->description('تنسيق عند انتهاء الكتابة + حفظ أعداد صحيحة')
                    ->icon('heroicon-o-banknotes')
                    ->iconColor('primary')
                    ->collapsible()
                    ->collapsed(false)
                    ->compact()
                    ->schema([

                        Hidden::make('invoice_subtotal')->dehydrated(true)->default(0)->dehydrateStateUsing(fn ($s) => self::toInt($s)),
                        Hidden::make('invoice_discount')->dehydrated(true)->default(0)->dehydrateStateUsing(fn ($s) => self::toInt($s)),
                        Hidden::make('invoice_shipping')->dehydrated(true)->default(0)->dehydrateStateUsing(fn ($s) => self::toInt($s)),
                        Hidden::make('invoice_paid')->dehydrated(true)->default(0)->dehydrateStateUsing(fn ($s) => self::toInt($s)),

                        Hidden::make('total_discount')->dehydrated(true)->default(0)->dehydrateStateUsing(fn ($s) => self::toInt($s)),
                        Hidden::make('total_shipping')->dehydrated(true)->default(0)->dehydrateStateUsing(fn ($s) => self::toInt($s)),
                        Hidden::make('grand_total')->dehydrated(true)->default(0)->dehydrateStateUsing(fn ($s) => self::toInt($s)),
                        Hidden::make('total_paid')->dehydrated(true)->default(0)->dehydrateStateUsing(fn ($s) => self::toInt($s)),
                        Hidden::make('total_due')->dehydrated(true)->default(0)->dehydrateStateUsing(fn ($s) => self::toInt($s)),

                        Grid::make(12)->schema([

                            TextInput::make('subtotal')
                                ->label('إجمالي المنتجات')
                                ->suffix('د.ع')
                                ->disabled()
                                ->dehydrated(true)
                                ->afterStateHydrated(function (Set $set, Get $get) {
                                    $set('subtotal', self::formatInt(self::calculateItemsTotal($get)));
                                    self::updateAllTotals($set, $get);
                                })
                                ->formatStateUsing(fn ($s) => self::formatInt($s))
                                ->dehydrateStateUsing(fn ($s) => self::toInt($s))
                                ->extraInputAttributes(self::NUM_ATTRS)
                                ->columnSpan(4),

                            self::moneyInput('shipping')
                                ->label('أجور التوصيل')
                                ->default(0)
                                ->minValue(0)
                                ->suffix('د.ع')
                                ->afterStateUpdated(fn (Set $set, Get $get) => self::updateAllTotals($set, $get))
                                ->columnSpan(4),

                            self::moneyInput('discount')
                                ->label('الخصم')
                                ->default(0)
                                ->minValue(0)
                                ->suffix('د.ع')
                                ->afterStateUpdated(fn (Set $set, Get $get) => self::updateAllTotals($set, $get))
                                ->columnSpan(4),

                            TextInput::make('net_total')
                                ->label('الصافي')
                                ->suffix('د.ع')
                                ->disabled()
                                ->dehydrated(true)
                                ->afterStateHydrated(function (Set $set, Get $get) {
                                    $set('net_total', self::formatInt(self::calculateNetTotal($get)));
                                    self::updateAllTotals($set, $get);
                                })
                                ->formatStateUsing(fn ($s) => self::formatInt($s))
                                ->dehydrateStateUsing(fn ($s) => self::toInt($s))
                                ->extraInputAttributes(self::NUM_ATTRS)
                                ->columnSpan(4),

                            self::moneyInput('paid_amount')
                                ->label('المدفوع')
                                ->default(0)
                                ->minValue(0)
                                ->suffix('د.ع')
                                ->afterStateUpdated(fn (Set $set, Get $get) => self::updateAllTotals($set, $get))
                                ->columnSpan(4),

                            TextInput::make('total_due')
                                ->label('المتبقي')
                                ->suffix('د.ع')
                                ->disabled()
                                ->dehydrated(true)
                                ->afterStateHydrated(function (Set $set, Get $get) {
                                    $set('total_due', self::formatInt(max(0, self::calculateNetTotal($get) - self::toInt(self::getRootScalar($get, 'paid_amount', 0)))));
                                    self::updateAllTotals($set, $get);
                                })
                                ->formatStateUsing(fn ($s) => self::formatInt($s))
                                ->dehydrateStateUsing(fn ($s) => self::toInt($s))
                                ->extraInputAttributes(self::NUM_ATTRS)
                                ->columnSpan(4),
                        ]),
                    ]),
            ]);
    }
}
