<?php

namespace App\Filament\Resources\Invoices\Pages;

use App\Filament\Resources\Pos\PosResource;
use App\Models\Branch;
use App\Models\ClientInstallmentPlan;
use App\Models\Invoice;
use App\Models\InvoiceItem;
use App\Models\StockBalance;
use App\Services\Accounting\ClientAccountingService;
use App\Services\Invoices\DeleteInvoiceService;
use App\Services\Invoices\UpdateInvoiceService;
use Filament\Actions\DeleteAction;
use Filament\Actions\ViewAction;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\EditRecord;
use Illuminate\Database\Eloquent\Model;
use Throwable;

class EditPos extends EditRecord
{
    protected static string $resource = PosResource::class;

    /* ============================ Helpers ============================ */

    private static function toInt(mixed $v): int
    {
        if ($v === null || $v === '') {
            return 0;
        }
        if (is_int($v)) {
            return $v;
        }
        if (is_float($v)) {
            return (int) $v;
        }

        $s = (string) $v;
        $s = strtr($s, [
            '٠' => '0',
            '١' => '1',
            '٢' => '2',
            '٣' => '3',
            '٤' => '4',
            '٥' => '5',
            '٦' => '6',
            '٧' => '7',
            '٨' => '8',
            '٩' => '9',
            '۰' => '0',
            '۱' => '1',
            '۲' => '2',
            '۳' => '3',
            '۴' => '4',
            '۵' => '5',
            '۶' => '6',
            '۷' => '7',
            '۸' => '8',
            '۹' => '9',
        ]);

        $s = preg_replace('/[^\d]/', '', $s);

        return (int) ($s === '' ? 0 : $s);
    }

    private function derivePaymentStatus(string $uiStatus, int $paid, int $due): string
    {
        if ($uiStatus === 'installment') {
            return 'installment';
        }
        if ($due <= 0) {
            return 'paid';
        }
        if ($paid > 0 && $due > 0) {
            return 'partial';
        }

        return 'unpaid';
    }

    /**
     * ✅ حساب وتثبيت المجاميع المالية سيرفر-سايد (INT فقط) قبل الحفظ
     * يعتمد فقط على ready_items + shipping + discount + paid_amount + min_paid_amount.
     */
    private function computeFinancialsForPosEdit(array $data): array
    {
        $ready = is_array($data['ready_items'] ?? null) ? $data['ready_items'] : [];

        $clean = [];
        $itemsSubtotal = 0;

        foreach ($ready as $row) {
            if (! is_array($row)) {
                continue;
            }

            $variantId = (int) ($row['product_variant_id'] ?? 0);
            if ($variantId <= 0) {
                continue;
            }

            $productId = (int) ($row['product_id'] ?? 0);

            $qty = self::toInt($row['quantity'] ?? 0);
            if ($qty <= 0) {
                $qty = 1;
            }

            $unitPrice = self::toInt($row['unit_price'] ?? 0);
            if ($unitPrice < 0) {
                $unitPrice = 0;
            }

            $lineTotal = $unitPrice * $qty;
            $itemsSubtotal += $lineTotal;

            $clean[] = array_merge($row, [
                'product_id' => $productId ?: null,
                'product_variant_id' => $variantId,
                'quantity' => $qty,
                'unit_price' => $unitPrice,
                'measurement' => 1,
                'line_total' => $lineTotal,
                'item_type' => 'product',
            ]);
        }

        $data['ready_items'] = array_values($clean);

        $shipping = self::toInt($data['shipping'] ?? 0);
        if ($shipping < 0) {
            $shipping = 0;
        }

        $discount = self::toInt($data['discount'] ?? 0);
        if ($discount < 0) {
            $discount = 0;
        }

        $net = max(0, $itemsSubtotal + $shipping - $discount);

        $minPaid = self::toInt($data['min_paid_amount'] ?? 0);
        if ($minPaid < 0) {
            $minPaid = 0;
        }

        $paidUi = self::toInt($data['paid_amount'] ?? 0);
        if ($paidUi < 0) {
            $paidUi = 0;
        }

        $paymentUi = (string) ($data['payment_status'] ?? 'paid');

        if ($paymentUi === 'paid') {
            $paid = max($net, $minPaid);
        } else {
            $paid = max($paidUi, $minPaid);
        }

        $paid = min($paid, $net);
        $due = max(0, $net - $paid);

        // ===== الفاتورة الأساسية =====
        $data['invoice_subtotal'] = $itemsSubtotal;
        $data['invoice_discount'] = $discount;
        $data['invoice_shipping'] = $shipping;
        $data['invoice_paid'] = $paid;

        // احتياطي
        $data['invoice_net_total'] = $net;
        $data['invoice_due'] = $due;

        // ===== المجاميع النهائية (combined) =====
        $data['subtotal'] = $itemsSubtotal;
        $data['total_discount'] = $discount;
        $data['total_shipping'] = $shipping;
        $data['grand_total'] = $net;
        $data['total_paid'] = $paid;
        $data['total_due'] = $due;

        // واجهة POS
        $data['paid_amount'] = $paid;
        $data['net_total'] = $net;
        $data['shipping'] = $shipping;
        $data['discount'] = $discount;

        // ✅ تثبيت payment_status بناء على الناتج النهائي
        $data['payment_status'] = $this->derivePaymentStatus($paymentUi, $paid, $due);

        return $data;
    }

    /* ============================ Fill ============================ */

    protected function mutateFormDataBeforeFill(array $data): array
    {
        /** @var Invoice $invoice */
        $invoice = $this->record->load('items');

        $items = $invoice->items ?? collect();

        /** @var ClientAccountingService $acc */
        $acc = app(ClientAccountingService::class);

        $currentPaidActual = (int) $acc->invoicePaidSum($invoice->fresh());

        // ✅ POS دائماً بيع جاهز
        $invoiceType = 'ready';

        $primaryWarehouseId = null;
        if ($invoice->branch_id) {
            $primaryWarehouse = Branch::find((int) $invoice->branch_id)?->primaryWarehouse()->first();
            $primaryWarehouseId = $primaryWarehouse?->id;
        }

        // 1) رأس الفاتورة
        $data['branch_id'] = (int) $invoice->branch_id;
        $data['user_id'] = (int) $invoice->user_id;
        $data['client_id'] = (int) $invoice->client_id;
        $data['invoice_number'] = (string) $invoice->invoice_number;

        $data['invoice_type'] = $invoiceType;
        $data['sale_type'] = (string) ($invoice->sale_type ?? 'retail');
        $data['invoice_date'] = optional($invoice->invoice_date)->toDateString();
        $data['notes'] = $invoice->notes;

        // ✅ الحد الأدنى للمدفوع هو المحصّل فعلياً
        $data['min_paid_amount'] = (int) $currentPaidActual;

        // 2) عناصر POS: products فقط
        $productItems = $items->where('item_type', InvoiceItem::TYPE_PRODUCT)->values();

        $data['ready_items'] = $productItems
            ->map(function (InvoiceItem $item) use ($primaryWarehouseId) {
                $availableQty = 0;

                if ($primaryWarehouseId && $item->product_variant_id) {
                    $balance = StockBalance::query()
                        ->where('warehouse_id', (int) $primaryWarehouseId)
                        ->where('product_variant_id', (int) $item->product_variant_id)
                        ->first();

                    if ($balance) {
                        $availableQty = max(0, (int) $balance->on_hand - (int) $balance->reserved_qty);
                    }
                }

                return [
                    'product_id' => (int) $item->product_id,
                    'product_variant_id' => (int) $item->product_variant_id,
                    'quantity' => (int) $item->quantity,
                    'unit_price' => (int) $item->unit_price,
                    'measurement' => 1,
                    'line_total' => (int) $item->line_total,
                    'item_type' => 'product',
                    'available_quantity' => (int) $availableQty,
                    'notes' => $item->notes,
                ];
            })
            ->toArray();

        // ✅ نلغي أي شيء غير POS
        $data['items'] = [];
        $data['accessory_items'] = [];

        // 3) مبالغ أولية للعرض
        $productsSubtotal = (int) $productItems->sum('line_total');

        $discount = (int) ($invoice->discount_total ?? 0);
        $shipping = (int) ($invoice->shipping_total ?? 0);

        $net = max($productsSubtotal + $shipping - $discount, 0);

        $paidFromDb = (int) ($invoice->paid_amount ?? 0);
        $paid = min($paidFromDb, $net);
        $paid = max($paid, $currentPaidActual);
        $paid = min($paid, $net);

        $due = max($net - $paid, 0);

        // ====== حقول الـ Services كما تتوقعها UpdateInvoiceService ======
        $data['invoice_subtotal'] = $productsSubtotal;
        $data['invoice_shipping'] = $shipping;
        $data['invoice_discount'] = $discount;
        $data['invoice_net_total'] = $net;
        $data['invoice_paid'] = $paid;
        $data['invoice_due'] = $due;

        $data['subtotal'] = $productsSubtotal;
        $data['total_discount'] = $discount;
        $data['total_shipping'] = $shipping;
        $data['grand_total'] = $net;

        $data['total_paid'] = (int) max($paid, $currentPaidActual);
        $data['total_due'] = $due;

        $data['shipping'] = $shipping;
        $data['discount'] = $discount;
        $data['paid_amount'] = $paid;
        $data['net_total'] = $net;

        // حالة الدفع + أقساط
        $plan = ClientInstallmentPlan::query()
            ->where('invoice_type', Invoice::class)
            ->where('invoice_id', $invoice->id)
            ->where('status', 'active')
            ->with('schedules')
            ->first();

        if ($plan) {
            $schedules = $plan->schedules()->orderBy('sequence')->get();

            $data['installments'] = $schedules
                ->map(fn ($schedule) => [
                    'sequence' => (int) $schedule->sequence,
                    'amount' => (int) $schedule->amount,
                    'due_date' => optional($schedule->due_date)->toDateString(),
                    'status' => $schedule->status ?? 'pending',
                ])
                ->values()
                ->toArray();

            $data['installments_count'] = count($data['installments']) ?: null;

            $data['payment_status'] = ((int) ($plan->remaining_amount ?? $plan->due_amount ?? 0) > 0)
                ? 'installment'
                : 'paid';
        } else {
            $data['installments'] = [];
            $data['installments_count'] = null;
            $data['payment_status'] = $this->derivePaymentStatus(
                (string) ($invoice->payment_status ?? 'paid'),
                (int) $paid,
                (int) $due
            );
        }

        return $data;
    }

    /* ============================ Save ============================ */

    protected function mutateFormDataBeforeSave(array $data): array
    {
        // ✅ حماية: رقم الفاتورة لا يتغير في التعديل
        unset($data['invoice_number']);

        // ✅ POS ثابت
        $data['invoice_type'] = 'ready';

        // ✅ منع الأقسام غير المدعومة
        $data['items'] = [];
        $data['accessory_items'] = [];

        // ✅ تنظيف ready_items: نحتفظ فقط بما فيه variant
        $data['ready_items'] = is_array($data['ready_items'] ?? null) ? $data['ready_items'] : [];
        $data['ready_items'] = array_values(array_filter($data['ready_items'], function ($row) {
            return is_array($row) && ! empty($row['product_variant_id']);
        }));

        if (count($data['ready_items']) === 0) {
            throw new \InvalidArgumentException('أضف عنصرًا واحدًا على الأقل وحدد المتغير والكمية.');
        }

        // ✅ تثبيت المجاميع المالية سيرفر-سايد
        $data = $this->computeFinancialsForPosEdit($data);

        // ✅ حماية: لا ترسل مدفوع أقل من المحصّل فعلياً
        $minPaid = self::toInt($data['min_paid_amount'] ?? 0);
        if ($minPaid > 0) {
            $data['paid_amount'] = max(self::toInt($data['paid_amount'] ?? 0), $minPaid);
            $data['total_paid'] = max(self::toInt($data['total_paid'] ?? 0), $minPaid);

            $data = $this->computeFinancialsForPosEdit($data);
        }

        // ✅ تثبيت payment_status بعد كل شيء
        $data['payment_status'] = $this->derivePaymentStatus(
            (string) ($data['payment_status'] ?? 'paid'),
            (int) ($data['total_paid'] ?? 0),
            (int) ($data['total_due'] ?? 0)
        );

        return $data;
    }

    /* ============================ Update ============================ */

    protected function handleRecordUpdate(Model $record, array $data): Model
    {
        /** @var UpdateInvoiceService $service */
        $service = app(UpdateInvoiceService::class);

        try {
            // ✅ ضمان إضافي
            unset($data['invoice_number']);

            $data['invoice_type'] = 'ready';
            $data['items'] = [];
            $data['accessory_items'] = [];

            // ✅ تثبيت payment_status أيضاً قبل الإرسال
            $data['payment_status'] = $this->derivePaymentStatus(
                (string) ($data['payment_status'] ?? 'paid'),
                (int) ($data['total_paid'] ?? 0),
                (int) ($data['total_due'] ?? 0)
            );

            return $service->updateFromForm($record, $data);
        } catch (Throwable $e) {
            report($e);

            Notification::make()
                ->title('حدث خطأ أثناء تحديث الفاتورة')
                ->body($e->getMessage())
                ->danger()
                ->persistent()
                ->send();

            throw $e;
        }
    }

    /* ============================ Header Actions ============================ */

    protected function getHeaderActions(): array
    {
        return [
            ViewAction::make(),

            DeleteAction::make()
                ->using(function (Model $record) {
                    /** @var DeleteInvoiceService $service */
                    $service = app(DeleteInvoiceService::class);

                    try {
                        $service->execute($record);
                    } catch (Throwable $e) {
                        report($e);

                        Notification::make()
                            ->title('حدث خطأ أثناء حذف الفاتورة')
                            ->body($e->getMessage())
                            ->danger()
                            ->persistent()
                            ->send();

                        throw $e;
                    }
                })
                ->successNotificationTitle('تم حذف الفاتورة وجميع تأثيراتها بنجاح'),
        ];
    }
}
