<?php

namespace App\Filament\Resources\Pos\Pages;

use App\Filament\Resources\Pos\PosResource;
use App\Models\ProductVariant;
use App\Services\Invoices\CreateInvoiceService;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\CreateRecord;
use Illuminate\Database\Eloquent\Model;
use Throwable;

class CreatePos extends CreateRecord
{
    protected static string $resource = PosResource::class;

    protected function mutateFormDataBeforeCreate(array $data): array
    {
        $data['branch_id'] = (int) ($data['branch_id'] ?? user_info('branch_id') ?? 0);
        $data['user_id'] = (int) ($data['user_id'] ?? user_info('id') ?? 0);

        if ($data['branch_id'] <= 0 || $data['user_id'] <= 0) {
            throw new \RuntimeException('تعذر تحديد الفرع أو المستخدم الحالي.');
        }

        // ✅ لا نسمح أبداً بتمرير رقم فاتورة من الواجهة (Preview فقط)
        unset($data['invoice_number']);

        // ✅ POS دائماً بيع جاهز
        $data['invoice_type'] = 'ready';

        // ✅ POS لا يدعم المخصص ولا الإكسسوارات
        $data['items'] = [];
        $data['accessory_items'] = [];

        // ✅ البنود الجاهزة فقط + تنظيف صارم (مع تثبيت السعر)
        $ready = is_array($data['ready_items'] ?? null) ? $data['ready_items'] : [];
        $ready = $this->sanitizeReadyRows($ready, (string) ($data['sale_type'] ?? 'retail'));

        if (count($ready) === 0) {
            throw new \InvalidArgumentException('أضف عنصرًا واحدًا على الأقل وحدد المتغير والكمية.');
        }

        $data['ready_items'] = $ready;

        // ✅ تثبيت كل المجاميع المالية سيرفر-سايد (INT فقط)
        $data = $this->computeFinancialsForReady($data);

        // ✅ تثبيت payment_status سيرفر-سايد
        $data['payment_status'] = $this->derivePaymentStatus(
            (string) ($data['payment_status'] ?? 'paid'),
            (int) ($data['total_paid'] ?? 0),
            (int) ($data['total_due'] ?? 0)
        );

        return $data;
    }

    private function computeFinancialsForReady(array $data): array
    {
        $ready = is_array($data['ready_items'] ?? null) ? $data['ready_items'] : [];
        $itemsSubtotal = 0;

        foreach ($ready as $row) {
            if (! is_array($row)) {
                continue;
            }

            $qty = $this->toInt($row['quantity'] ?? 0);
            $price = $this->toInt($row['unit_price'] ?? 0);

            if ($qty > 0 && $price > 0) {
                $itemsSubtotal += ($qty * $price);
            }
        }

        $shipping = $this->toInt($data['shipping'] ?? 0);
        $discount = $this->toInt($data['discount'] ?? 0);

        $net = max(0, $itemsSubtotal + $shipping - $discount);

        $paid = $this->toInt($data['paid_amount'] ?? 0);
        $minPaid = $this->toInt($data['min_paid_amount'] ?? 0);

        $paymentUi = (string) ($data['payment_status'] ?? 'paid');

        // ✅ paid: يجب أن يكون مساوٍ للصافي (لكن لا ينزل عن المحصّل فعلياً)
        if ($paymentUi === 'paid') {
            $paid = max($net, $minPaid);
        } else {
            // installment / unpaid / partial ... إلخ
            $paid = max($paid, $minPaid);
        }

        $paid = min($paid, $net);
        $due = max(0, $net - $paid);

        // ===== حقول الخدمة =====
        $data['invoice_subtotal'] = $itemsSubtotal;
        $data['invoice_discount'] = $discount;
        $data['invoice_shipping'] = $shipping;
        $data['invoice_paid'] = $paid;

        // ===== حقول الفاتورة =====
        $data['subtotal'] = $itemsSubtotal;
        $data['total_discount'] = $discount;
        $data['total_shipping'] = $shipping;
        $data['grand_total'] = $net;
        $data['total_paid'] = $paid;
        $data['total_due'] = $due;

        // واجهة POS
        $data['net_total'] = $net;
        $data['paid_amount'] = $paid;
        $data['shipping'] = $shipping;
        $data['discount'] = $discount;

        return $data;
    }

    private function sanitizeReadyRows(array $rows, string $saleType): array
    {
        $filtered = [];
        $saleType = in_array($saleType, ['retail', 'wholesale', 'agent'], true) ? $saleType : 'retail';

        foreach ($rows as $idx => $row) {
            if (! is_array($row)) {
                continue;
            }

            $variantId = (int) ($row['product_variant_id'] ?? 0);
            if ($variantId <= 0) {
                continue;
            }

            $qty = $this->toInt($row['quantity'] ?? 0);
            if ($qty <= 0) {
                $qty = 1;
            }

            $unitPrice = $this->toInt($row['unit_price'] ?? 0);
            if ($unitPrice <= 0) {
                $unitPrice = $this->getVariantUnitPrice($variantId, $saleType);
            }

            if ($unitPrice <= 0) {
                $n = $idx + 1;
                throw new \InvalidArgumentException(
                    "العنصر رقم {$n}: لا يوجد سعر لهذا المتغير. يرجى إضافة سعر أو أدخل سعر القطعة يدوياً."
                );
            }

            $row['product_variant_id'] = $variantId;
            $row['quantity'] = $qty;
            $row['unit_price'] = $unitPrice;

            $row['line_total'] = $this->toInt($row['line_total'] ?? ($unitPrice * $qty));
            $row['measurement'] = 1;
            $row['item_type'] = $row['item_type'] ?? 'product';

            $filtered[] = $row;
        }

        return array_values($filtered);
    }

    private function getVariantUnitPrice(int $variantId, string $saleType): int
    {
        if ($variantId <= 0) {
            return 0;
        }

        $saleType = in_array($saleType, ['retail', 'wholesale', 'agent'], true) ? $saleType : 'retail';

        /** @var ProductVariant|null $variant */
        $variant = ProductVariant::query()->find($variantId);
        if (! $variant) {
            return 0;
        }

        return max(0, (int) $variant->getPriceForType($saleType));
    }

    private function derivePaymentStatus(string $uiStatus, int $paid, int $due): string
    {
        // ✅ إذا المستخدم اختار أقساط نخليها أقساط
        if ($uiStatus === 'installment') {
            return 'installment';
        }

        if ($due <= 0) {
            return 'paid';
        }

        if ($paid > 0 && $due > 0) {
            return 'partial';
        }

        return 'unpaid';
    }

    private function toInt(mixed $v): int
    {
        if ($v === null || $v === '') {
            return 0;
        }
        if (is_int($v)) {
            return $v;
        }

        $s = (string) $v;
        $s = strtr($s, [
            '٠' => '0',
            '١' => '1',
            '٢' => '2',
            '٣' => '3',
            '٤' => '4',
            '٥' => '5',
            '٦' => '6',
            '٧' => '7',
            '٨' => '8',
            '٩' => '9',
            '۰' => '0',
            '۱' => '1',
            '۲' => '2',
            '۳' => '3',
            '۴' => '4',
            '۵' => '5',
            '۶' => '6',
            '۷' => '7',
            '۸' => '8',
            '۹' => '9',
        ]);

        $s = preg_replace('/[^\d]/', '', $s);

        return (int) ($s === '' ? 0 : $s);
    }

    protected function handleRecordCreation(array $data): Model
    {
        /** @var CreateInvoiceService $service */
        $service = app(CreateInvoiceService::class);

        try {
            // ✅ ضمان إضافي: لا نسمح بتمرير invoice_number
            unset($data['invoice_number']);

            // ✅ POS ثابت
            $data['invoice_type'] = 'ready';
            $data['items'] = [];
            $data['accessory_items'] = [];

            // ✅ تأكيد payment_status سيرفر-سايد قبل الإرسال للخدمة
            $data['payment_status'] = $this->derivePaymentStatus(
                (string) ($data['payment_status'] ?? 'paid'),
                (int) ($data['total_paid'] ?? 0),
                (int) ($data['total_due'] ?? 0)
            );

            $record = $service->createFromForm($data);

            Notification::make()
                ->title('تم إنشاء بيع مباشر بنجاح')
                ->success()
                ->send();

            return $record;
        } catch (Throwable $e) {
            report($e);

            Notification::make()
                ->title('حدث خطأ أثناء إنشاء البيع المباشر')
                ->body($e->getMessage())
                ->danger()
                ->persistent()
                ->send();

            throw $e;
        }
    }

    protected function getRedirectUrl(): string
    {
        return $this->getResource()::getUrl('view', ['record' => $this->record]);
    }
}
