<?php

namespace App\Filament\Resources\Permissions\Tables;

use Filament\Tables\Table;
use App\Enums\RoleEnum;
use Filament\Actions\EditAction;
use App\Support\PermissionsLabel;
use Filament\Actions\ActionGroup;
use Filament\Actions\DeleteAction;
use Spatie\Permission\Models\Role;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Illuminate\Database\Eloquent\Builder;

class PermissionsTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->columns([
                // عرض الاسم مترجمًا بالعربية
                TextColumn::make('name')
                    ->label('الصلاحية')
                    ->formatStateUsing(fn(string $state) => PermissionsLabel::label($state))
                    ->searchable()
                    ->sortable(),
                // الأدوار المالكة لهذه الصلاحية
                TextColumn::make('roles.name')
                    ->label('الدور')
                    ->badge()
                    ->separator(', ')
                    ->color(function ($state) {
                        $enum = RoleEnum::tryFrom($state);
                        return $enum?->getColor() ?? 'gray';
                    })
                    ->formatStateUsing(fn(string $state) => __("roles.$state")) // ⬅ ترجمة
                    ->searchable()
                    ->sortable(),

            ])
            ->filters([
                SelectFilter::make('role')
                    ->label('الدور')
                    ->multiple()                 // ← يسمح باختيار أكثر من دور
                    ->options(function () {
                        return Role::query()
                            ->where('name', '!=', 'Administrator')
                            ->pluck('name', 'name')
                            ->mapWithKeys(fn($name) => [$name => __('roles.' . $name)])
                            ->all();
                    })
                    ->searchable()
                    ->query(function (Builder $query, array $data) {
                        $values = (array) ($data['values'] ?? $data['value'] ?? []);
                        if ($values) {
                            $query->whereHas('roles', fn($q) => $q->whereIn('name', $values));
                        }
                    }),
            ])
            ->recordActions([
                ActionGroup::make([
                    EditAction::make()
                        ->label('تعديل')
                        ->icon('heroicon-m-pencil')
                        ->color('gray'),

                    DeleteAction::make()
                        ->label('حذف')
                        ->icon('heroicon-m-trash')
                        ->color('gray')
                        ->requiresConfirmation(),
                ])
                    ->label('') // نخليها أيقونة فقط
                    ->icon('heroicon-m-ellipsis-vertical')
                    ->iconButton()
                    ->color('gray')
                    ->size('lg')
                    ->extraAttributes(['class' => 'fi-dd-trigger']),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ]);
    }
}
