<?php

namespace App\Filament\Resources\Payroll\PayrollRuns\Tables;

use Filament\Tables\Table;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TrashedFilter;

use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\IconColumn;

use Filament\Actions\EditAction;
use Filament\Actions\ViewAction;

use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\ForceDeleteBulkAction;
use Filament\Actions\RestoreBulkAction;

use Illuminate\Database\Eloquent\Builder;

class PayrollRunsTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->columns([
                // الفترة بصيغة 2025/03
                TextColumn::make('label')
                    ->label('الفترة')
                    ->getStateUsing(fn($record) => $record->label)
                    ->searchable(query: function (Builder $query, string $search): Builder {
                        // يسمح بالبحث عن "2025" أو "03" أو "2025/03"
                        $search = trim($search);
                        return $query->where('year', 'like', "%{$search}%")
                            ->orWhere('month', 'like', "%{$search}%");
                    })
                    ->sortable(),

                // (اختياري) الفرع — احذفه إذا لا تريد إظهاره
                TextColumn::make('branch.name')
                    ->label('الفرع')
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->sortable()
                    ->searchable(),

                // من قام بالعملية
                TextColumn::make('user.name')
                    ->label('المستخدم')
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->sortable()
                    ->searchable(),

                // حالة الإغلاق
                IconColumn::make('is_locked')
                    ->label('مقفلة')
                    ->boolean()
                    ->sortable(),

                TextColumn::make('locked_at')
                    ->label('تاريخ الإغلاق')
                    ->dateTime('Y-m-d H:i')
                    ->placeholder('—')
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->sortable(),

                // عدد الرواتب داخل الدفعة
                TextColumn::make('payrolls_count')
                    ->label('عدد الرواتب')
                    ->counts('payrolls')
                    ->alignCenter()
                    ->sortable(),

                // مجموع الصافي داخل الدفعة (requires withSum)
                TextColumn::make('payrolls_sum_net_salary')
                    ->label('مجموع الصافي')
                    ->getStateUsing(fn($record) => (int) ($record->payrolls_sum_net_salary ?? 0))
                    ->formatStateUsing(fn($state) => number_format((int) $state))
                    ->alignEnd()
                    ->sortable(),

                TextColumn::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->dateTime('Y-m-d H:i')
                    ->since()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->sortable(),
            ])
            ->modifyQueryUsing(function (Builder $query) {
                // لتحميل العدّ والمجموع بدون N+1
                return $query->withCount('payrolls')
                    ->withSum('payrolls', 'net_salary');
            })
            ->defaultSort('year', 'desc')
            ->filters([
                // فلتر سنة
                SelectFilter::make('year')
                    ->label('السنة')
                    ->options(function (): array {
                        $y = (int) now()->year;
                        $years = range($y - 5, $y + 1);

                        return collect($years)
                            ->mapWithKeys(fn($v) => [$v => (string) $v])
                            ->toArray();
                    }),

                // فلتر شهر
                SelectFilter::make('month')
                    ->label('الشهر')
                    ->options(collect(range(1, 12))->mapWithKeys(fn($m) => [
                        $m => sprintf('%02d', $m),
                    ])->toArray()),

                // فلتر حالة الإغلاق
                Filter::make('locked')
                    ->label('المقفلة فقط')
                    ->query(fn(Builder $query) => $query->where('is_locked', true)),

                Filter::make('unlocked')
                    ->label('غير المقفلة فقط')
                    ->query(fn(Builder $query) => $query->where('is_locked', false)),

                // سلة المحذوفات (SoftDeletes)
                TrashedFilter::make()
                    ->label('المحذوفات'),
            ])
            ->recordActions([
                ViewAction::make(),

                EditAction::make()
                    ->visible(fn($record) => ! (bool) $record->is_locked),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                ]),
            ]);
    }
}
