<?php

namespace App\Filament\Resources\PayGrades;

use BackedEnum;
use App\Models\PayGrade;
use App\Support\NavGroups;
use Filament\Tables\Table;
use Filament\Schemas\Schema;
use Filament\Resources\Resource;
use Filament\Support\Icons\Heroicon;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use App\Filament\Resources\PayGrades\Pages\ManagePayGrades;
use App\Filament\Resources\PayGrades\Tables\PayGradesTable;

class PayGradeResource extends Resource
{
    protected static ?string $model = PayGrade::class;
    protected static string|\UnitEnum|null $navigationGroup = NavGroups::MEMBERS;
    protected static string|BackedEnum|null $navigationIcon = Heroicon::Wallet;
    protected static ?string $navigationLabel               = 'سُلَّم الرواتب';
    protected static ?string $modelLabel                    = 'درجة وظيفية';
    protected static ?string $pluralModelLabel              = 'سُلَّم الرواتب';
    protected static ?int $navigationSort                   = 13;
    protected static ?string $recordTitleAttribute          = 'name';

    public static function form(Schema $form): Schema
    {
        return $form->schema([
            TextInput::make('name')
                ->label('الوظيفة')
                ->required()
                ->maxLength(100),

            // حقل العرض فقط
            TextInput::make('amount')
                ->label('الراتب')
                ->required()
                ->type('text') // نستخدم نص حتى نقدر نضع الفواصل بصرياً
                // عرض القيمة منسّقة عند التحميل/التحرير:
                ->formatStateUsing(fn($state) => $state !== null ? number_format((int) $state) : '')
                // عند الحفظ تُحوَّل لأي مدخلات (بما فيها أرقام عربية) إلى رقم صحيح بدون فواصل:
                ->dehydrateStateUsing(function ($state) {
                    // حوّل الأرقام العربية-الهندية إلى لاتينية
                    $arabic = ['٠', '١', '٢', '٣', '٤', '٥', '٦', '٧', '٨', '٩', '٫', '٬'];
                    $latin  = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '.', ','];
                    $str = strtr((string) $state, array_combine($arabic, $latin));

                    // احذف أي شيء غير رقم
                    $digitsOnly = preg_replace('/\D+/', '', $str) ?: '0';

                    return (int) $digitsOnly; // هذا الذي سيُخزَّن في عمود INT
                })
                // تنسيق حي أثناء الكتابة باستخدام Alpine:
                ->extraAlpineAttributes([
                    'x-data' => <<<'ALPINE'
                            {
                                format() {
                                    const raw = $el.value
                                        .replace(/[٠-٩]/g, d => '٠١٢٣٤٥٦٧٨٩'.indexOf(d)) // تحويل عربي->لاتيني
                                        .replace(/[^\d]/g, '');
                                    $el.value = raw ? new Intl.NumberFormat('en-US').format(Number(raw)) : '';
                                }
                            }
                        ALPINE,
                    'x-on:input' => 'format()',
                    'x-on:blur'  => 'format()',
                ])
                ->extraAttributes(['inputmode' => 'numeric'])
                ->suffix(' د.ع'),

            Select::make('is_active')
                ->label('الحالة')
                ->options([1 => 'مفعّل', 0 => 'متوقف'])
                ->visibleOn('edit')
                ->required()
                ->afterStateHydrated(function (Select $component, $state) {
                    $component->state($state ? 1 : 0);
                })
                ->dehydrateStateUsing(fn($state) => (bool) $state),
        ]);
    }


    public static function table(Table $table): Table
    {
        return PayGradesTable::configure($table);
    }

    public static function getPages(): array
    {
        return [
            'index' => ManagePayGrades::route('/'),
        ];
    }
}
