<?php

namespace App\Filament\Resources\Orders\Widgets;

use App\Enums\OrderStatus;
use App\Models\Order;
use Carbon\Carbon;
use Carbon\CarbonPeriod;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Select;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Schema;
use Filament\Widgets\StatsOverviewWidget as BaseWidget;
use Filament\Widgets\StatsOverviewWidget\Stat;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\HtmlString;

class OrdersStatsOverview extends BaseWidget implements HasForms
{
    use InteractsWithForms;

    protected static ?int $sort = 1;

    protected string $view = 'filament.widgets.orders-stats';

    public ?array $data = [];

    public function mount(): void
    {
        $this->form->fill([
            'period' => 'today',
            'from_date' => null,
            'to_date' => null,
        ]);
    }

    /* =========================
     * Helpers
     * ========================= */

    protected function fv(string $key, mixed $default = null): mixed
    {
        if (is_array($this->data) && array_key_exists($key, $this->data) && $this->data[$key] !== null) {
            return $this->data[$key];
        }

        return $default;
    }

    protected function periodValue(): string
    {
        return (string) ($this->fv('period') ?? 'today');
    }

    protected function fromValue(): mixed
    {
        return $this->fv('from_date');
    }

    protected function toValue(): mixed
    {
        return $this->fv('to_date');
    }

    public function hasActiveSelection(): bool
    {
        $period = $this->periodValue();

        if ($period !== 'today') {
            return true;
        }

        if ($period === 'custom' && ($this->fromValue() || $this->toValue())) {
            return true;
        }

        return false;
    }

    /* =========================
     * Form
     * ========================= */

    public function form(Schema $form): Schema
    {
        return $form
            ->schema([
                Grid::make(4)->schema([
                    Select::make('period')
                        ->label('اختر الفترة')
                        ->options([
                            'today' => 'اليوم',
                            'yesterday' => 'أمس',
                            'this_week' => 'الأسبوع الحالي',
                            'last_week' => 'الأسبوع الماضي',
                            'last_7_days' => 'آخر 7 أيام',
                            'this_month' => 'الشهر الحالي',
                            'last_month' => 'الشهر الماضي',
                            'last_30_days' => 'آخر 30 يوم',
                            'this_year' => 'هذه السنة',
                            'custom' => 'مخصص',
                            'all' => 'كل الفترات',
                        ])
                        ->default('today')
                        ->native(false)
                        ->reactive()
                        ->afterStateUpdated(function ($state) {
                            if ($state !== 'custom') {
                                $this->data['from_date'] = null;
                                $this->data['to_date'] = null;
                            }
                            $this->dispatch('$refresh');
                        }),

                    DatePicker::make('from_date')
                        ->label('من تاريخ')
                        ->visible(fn ($get) => $get('period') === 'custom')
                        ->native(false)
                        ->maxDate(fn ($get) => $get('to_date') ?: now())
                        ->reactive()
                        ->afterStateUpdated(fn () => $this->dispatch('$refresh')),

                    DatePicker::make('to_date')
                        ->label('إلى تاريخ')
                        ->visible(fn ($get) => $get('period') === 'custom')
                        ->native(false)
                        ->minDate(fn ($get) => $get('from_date'))
                        ->reactive()
                        ->afterStateUpdated(fn () => $this->dispatch('$refresh')),
                ]),
            ])
            ->statePath('data');
    }

    public function resetFilters(): void
    {
        $this->form->fill([
            'period' => 'today',
            'from_date' => null,
            'to_date' => null,
        ]);
        $this->dispatch('$refresh');
    }

    /* =========================
     * Date Range
     * ========================= */

    protected function resolveRange(): array
    {
        $period = $this->periodValue();
        $now = now();

        return match ($period) {
            'today' => [$now->copy()->startOfDay(), $now->copy()->endOfDay()],
            'yesterday' => [$now->copy()->subDay()->startOfDay(), $now->copy()->subDay()->endOfDay()],
            'this_week' => [$now->copy()->startOfWeek(), $now->copy()->endOfWeek()],
            'last_week' => [$now->copy()->subWeek()->startOfWeek(), $now->copy()->subWeek()->endOfWeek()],
            'last_7_days' => [$now->copy()->subDays(6)->startOfDay(), $now->copy()->endOfDay()],
            'this_month' => [$now->copy()->startOfMonth(), $now->copy()->endOfMonth()],
            'last_month' => [$now->copy()->subMonth()->startOfMonth(), $now->copy()->subMonth()->endOfMonth()],
            'last_30_days' => [$now->copy()->subDays(29)->startOfDay(), $now->copy()->endOfDay()],
            'this_year' => [$now->copy()->startOfYear(), $now->copy()->endOfYear()],
            'custom' => [
                $this->fromValue()
                    ? Carbon::parse($this->fromValue())->startOfDay()
                    : $now->copy()->startOfDay(),
                $this->toValue()
                    ? Carbon::parse($this->toValue())->endOfDay()
                    : $now->copy()->endOfDay(),
            ],
            'all' => [null, null],
            default => [$now->copy()->startOfDay(), $now->copy()->endOfDay()],
        };
    }

    /* =========================
     * Chart Data per Status
     * ========================= */

    protected function getChartForStatus(OrderStatus $status, int $days = 8): array
    {
        [$from, $to] = $this->resolveRange();
        $end = $to ?: now();
        $start = $from ?: $end->copy()->subDays($days - 1);

        $rows = Order::query()
            ->where('status', $status)
            ->whereBetween('created_at', [$start, $end])
            ->selectRaw('DATE(created_at) as d, COUNT(*) as c')
            ->groupBy('d')
            ->pluck('c', 'd')
            ->all();

        $out = [];
        foreach (CarbonPeriod::create($start->toDateString(), $end->toDateString()) as $day) {
            $k = Carbon::parse($day)->toDateString();
            $out[] = (int) ($rows[$k] ?? 0);
        }

        return $out;
    }

    /* =========================
     * Helpers
     * ========================= */

    protected function formatNumber(int|float $value): string
    {
        return number_format((int) $value, 0, '.', ',');
    }

    protected function hoverCardClasses(): string
    {
        return 'transition-all duration-200 ease-out hover:-translate-y-0.5 hover:shadow-xl dark:hover:shadow-2xl';
    }

    protected function animatedAmount(int $amount, string $key): HtmlString
    {
        return new HtmlString(
            $this->countUpSpan($amount, $key).
            '<span class="ms-1 text-sm font-medium text-gray-500 dark:text-gray-400">د.ع</span>'
        );
    }

    protected function animatedNumber(int $number, string $key): HtmlString
    {
        return new HtmlString($this->countUpSpan($number, $key));
    }

    protected function countUpSpan(int $to, string $key): string
    {
        $value = (int) $to;
        $abs = abs($value);
        $period = $this->periodValue();
        $wireKey = "stat-{$key}-{$period}-{$value}";
        $sign = $value < 0 ? '-' : '';

        return <<<HTML
<span
    wire:key="{$wireKey}"
    class="tabular-nums"
    x-data="{
        to: {$abs},
        v: 0,
        dur: 780,
        start() {
            const start = performance.now();
            const from = 0;
            const to = this.to;

            const tick = (t) => {
                const p = Math.min(1, (t - start) / this.dur);
                const eased = 1 - Math.pow(1 - p, 3);
                this.v = Math.round(from + (to - from) * eased);
                if (p < 1) requestAnimationFrame(tick);
            };

            requestAnimationFrame(tick);
        },
        fmt(n) { return new Intl.NumberFormat('en-US').format(n); }
    }"
    x-init="start()"
    x-text="'{$sign}' + fmt(v)"
></span>
HTML;
    }

    /* =========================
     * Stats Cards
     * ========================= */

    protected function getStats(): array
    {
        [$from, $to] = $this->resolveRange();

        $query = Order::query()
            ->when($from && $to, fn ($q) => $q->whereBetween('created_at', [$from, $to]));

        $statusCounts = (clone $query)
            ->select('status', DB::raw('COUNT(*) as count'), DB::raw('SUM(grand_total) as total'), DB::raw('SUM(COALESCE(paid_amount, 0)) as paid'))
            ->groupBy('status')
            ->get()
            ->keyBy('status');

        $statuses = [
            OrderStatus::Pending,
            OrderStatus::Confirmed,
            OrderStatus::Processing,
            OrderStatus::ReadyToShip,
            OrderStatus::Shipped,
            OrderStatus::Delivered,
            OrderStatus::Completed,
            OrderStatus::Canceled,
            OrderStatus::Refunded,
        ];

        $cardHover = $this->hoverCardClasses();
        $stats = [];

        foreach ($statuses as $status) {
            $row = $statusCounts->get($status->value);
            $count = (int) ($row?->count ?? 0);
            $total = (int) ($row?->total ?? 0);
            $paid = (int) ($row?->paid ?? 0);

            $chart = $this->getChartForStatus($status);

            $desc = 'عدد: '.$count.' | إجمالي: '.$this->formatNumber($total).' د.ع';
            if ($paid > 0) {
                $desc .= ' | مدفوع: '.$this->formatNumber($paid).' د.ع';
            }

            $stats[] = Stat::make($status->label(), $this->animatedNumber($count, $status->value))
                ->description($desc)
                ->descriptionIcon($status->icon())
                ->chart($chart)
                ->color($count > 0 ? $status->color() : 'gray')
                ->extraAttributes(['class' => $cardHover]);
        }

        return $stats;
    }

    protected function getColumns(): int
    {
        return 3;
    }
}
