<?php

namespace App\Filament\Resources\Orders\Tables;

use App\Enums\OrderStatus;
use App\Enums\PaymentStatus;
use App\Filament\Resources\Orders\OrderResource;
use App\Models\Order;
use App\Services\Orders\OrderService;
use Coolsam\Flatpickr\Forms\Components\Flatpickr;
use Filament\Actions\ActionGroup;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Actions\ViewAction;
use Filament\Notifications\Notification;
use Filament\Tables\Columns\SelectColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Carbon;

class OrdersTable
{
    private static array $oldStatuses = [];

    public static function configure(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('order_number')
                    ->label('رقم الطلب')
                    ->searchable()
                    ->sortable()
                    ->weight('bold')
                    ->copyable()
                    ->copyMessage('تم نسخ رقم الطلب'),

                TextColumn::make('client.name')
                    ->label('العميل')
                    ->searchable()
                    ->sortable()
                    ->description(fn (Order $record) => $record->client?->phone),

                SelectColumn::make('status')
                    ->label('حالة الطلب')
                    ->options(function (Order $record): array {
                        // حفظ الحالة القديمة
                        self::$oldStatuses[$record->id] = $record->status;

                        // الحصول على الانتقالات المتاحة من OrderService
                        $service = app(OrderService::class);
                        $transitions = $service->getAvailableTransitions($record);

                        // بناء قائمة الخيارات: الحالة الحالية + الانتقالات المتاحة
                        $options = [$record->status->value => $record->status->label()];

                        foreach ($transitions as $transition) {
                            $options[$transition['status']->value] = $transition['status']->label();
                        }

                        return $options;
                    })
                    ->selectablePlaceholder(false)
                    ->sortable()
                    ->afterStateUpdated(function (Order $record, $state) {
                        try {
                            $newStatus = OrderStatus::from($state);
                            $oldStatus = self::$oldStatuses[$record->id] ?? $record->status;

                            // إذا اختار نفس الحالة، لا نفعل شيء
                            if ($newStatus === $oldStatus) {
                                return;
                            }

                            $service = app(OrderService::class);

                            // استعادة الحالة القديمة أولاً
                            $record->updateQuietly(['status' => $oldStatus->value]);

                            // محاولة تنفيذ الانتقال عبر OrderService
                            $result = $service->transition($record->fresh(), $newStatus);

                            if ($result) {
                                Notification::make()
                                    ->success()
                                    ->title('تم التحديث')
                                    ->body('تم تحديث حالة الطلب إلى: '.$newStatus->label())
                                    ->send();
                            } else {
                                Notification::make()
                                    ->danger()
                                    ->title('خطأ')
                                    ->body('لا يمكن الانتقال إلى هذه الحالة')
                                    ->send();
                            }

                            // مسح الحالة المحفوظة
                            unset(self::$oldStatuses[$record->id]);
                        } catch (\Exception $e) {
                            // استعادة الحالة في حالة الخطأ
                            if (isset(self::$oldStatuses[$record->id])) {
                                $record->updateQuietly(['status' => self::$oldStatuses[$record->id]->value]);
                                unset(self::$oldStatuses[$record->id]);
                            }

                            Notification::make()
                                ->danger()
                                ->title('خطأ')
                                ->body('حدث خطأ: '.$e->getMessage())
                                ->send();
                        }
                    }),

                TextColumn::make('payment_status')
                    ->label('حالة الدفع')
                    ->badge()
                    ->formatStateUsing(fn (PaymentStatus $state) => $state->label())
                    ->color(fn (PaymentStatus $state) => $state->color())
                    ->sortable(),

                TextColumn::make('items_count')
                    ->label('العناصر')
                    ->counts('items')
                    ->badge()
                    ->color('gray')
                    ->alignCenter(),

                TextColumn::make('subtotal')
                    ->label('المجموع الفرعي')
                    ->formatStateUsing(fn ($state) => number_format($state, 0, '', ',').' د.ع')
                    ->alignEnd()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('discount_total')
                    ->label('الخصم')
                    ->formatStateUsing(fn ($state) => $state > 0 ? '-'.number_format($state, 0, '', ',').' د.ع' : '-')
                    ->color('danger')
                    ->alignEnd()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('grand_total')
                    ->label('الإجمالي')
                    ->formatStateUsing(fn ($state) => number_format($state, 0, '', ',').' د.ع')
                    ->weight('bold')
                    ->color('success')
                    ->alignEnd()
                    ->sortable(),

                TextColumn::make('branch.name')
                    ->label('الفرع')
                    ->badge()
                    ->color('info')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->dateTime('Y-m-d h:i A')
                    ->sortable()
                    ->toggleable(),

                TextColumn::make('confirmed_at')
                    ->label('تاريخ التأكيد')
                    ->dateTime('Y-m-d h:i A')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                SelectFilter::make('status')
                    ->label('حالة الطلب')
                    ->options(OrderStatus::options())
                    ->multiple(),

                SelectFilter::make('payment_status')
                    ->label('حالة الدفع')
                    ->options(collect(PaymentStatus::cases())->mapWithKeys(fn ($s) => [$s->value => $s->label()])->all())
                    ->multiple(),

                SelectFilter::make('branch_id')
                    ->label('الفرع')
                    ->relationship('branch', 'name')
                    ->multiple()
                    ->preload(),

                Filter::make('pending')
                    ->label('قيد الانتظار')
                    ->query(fn (Builder $query) => $query->where('status', OrderStatus::Pending)),

                Filter::make('today')
                    ->label('طلبات اليوم')
                    ->query(fn (Builder $query) => $query->whereDate('created_at', today())),

                Filter::make('date_range')
                    ->form([
                        Flatpickr::make('from_date')
                            ->label('من تاريخ')
                            ->displayFormat('Y-m-d')
                            ->native(false),
                        Flatpickr::make('to_date')
                            ->label('إلى تاريخ')
                            ->displayFormat('Y-m-d')
                            ->native(false),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['from_date'] ?? null,
                                fn (Builder $q, $date) => $q->whereDate('created_at', '>=', Carbon::parse($date))
                            )
                            ->when(
                                $data['to_date'] ?? null,
                                fn (Builder $q, $date) => $q->whereDate('created_at', '<=', Carbon::parse($date))
                            );
                    })
                    ->indicateUsing(function (array $data): array {
                        $indicators = [];
                        if ($data['from_date'] ?? null) {
                            $indicators['from_date'] = 'من: '.$data['from_date'];
                        }
                        if ($data['to_date'] ?? null) {
                            $indicators['to_date'] = 'إلى: '.$data['to_date'];
                        }

                        return $indicators;
                    }),
            ])
            ->recordActions([
                ActionGroup::make([
                    ViewAction::make(),
                    EditAction::make(),
                    DeleteAction::make()
                        ->visible(fn (Order $record) => $record->status === OrderStatus::Pending)
                        ->requiresConfirmation()
                        ->modalHeading('حذف الطلب')
                        ->modalDescription('هل أنت متأكد من حذف هذا الطلب؟'),
                ]),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make()
                        ->requiresConfirmation()
                        ->modalHeading('حذف الطلبات المحددة')
                        ->modalDescription('هل أنت متأكد من حذف الطلبات المحددة؟'),
                ]),
            ])
            ->defaultSort('created_at', 'desc')
            ->recordUrl(fn (Order $record) => OrderResource::getUrl('edit', ['record' => $record]))
            ->emptyStateHeading('لا توجد طلبات')
            ->emptyStateDescription('ابدأ بإضافة طلب جديد')
            ->emptyStateIcon('heroicon-o-shopping-cart')
            ->striped();
    }
}
