<?php

namespace App\Filament\Resources\Orders\Schemas;

use App\Enums\OrderStatus;
use App\Enums\PaymentStatus;
use App\Models\Order;
use Filament\Infolists\Components\RepeatableEntry;
use Filament\Infolists\Components\TextEntry;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;

class OrderInfolist
{
    public static function configure(Schema $schema): Schema
    {
        return $schema->components([
            self::sectionBasicInfo(),
            self::sectionOrderItems(),
            self::sectionTotals(),
            self::sectionShipping(),
            self::sectionNotes(),
            self::sectionTimestamps(),
        ]);
    }

    /* ================== المعلومات الأساسية ================== */

    protected static function sectionBasicInfo(): Section
    {
        return Section::make('معلومات الطلب')
            ->icon('heroicon-o-shopping-cart')
            ->columns(4)
            ->childComponents([
                TextEntry::make('order_number')
                    ->label('رقم الطلب')
                    ->weight('bold')
                    ->copyable()
                    ->copyMessage('تم نسخ رقم الطلب'),

                TextEntry::make('client.name')
                    ->label('العميل')
                    ->url(fn (Order $record) => $record->client_id
                        ? route('filament.admin.resources.clients.edit', $record->client_id)
                        : null),

                TextEntry::make('branch.name')
                    ->label('الفرع')
                    ->badge()
                    ->color('info'),

                TextEntry::make('currency.name')
                    ->label('العملة'),

                TextEntry::make('status')
                    ->label('حالة الطلب')
                    ->badge()
                    ->formatStateUsing(fn (OrderStatus $state) => $state->label())
                    ->color(fn (OrderStatus $state) => $state->color())
                    ->icon(fn (OrderStatus $state) => $state->icon()),

                TextEntry::make('payment_status')
                    ->label('حالة الدفع')
                    ->badge()
                    ->formatStateUsing(fn (PaymentStatus $state) => $state->label())
                    ->color(fn (PaymentStatus $state) => $state->color()),
            ]);
    }

    /* ================== عناصر الطلب ================== */

    protected static function sectionOrderItems(): Section
    {
        return Section::make('عناصر الطلب')
            ->icon('heroicon-o-cube')
            ->childComponents([
                RepeatableEntry::make('items')
                    ->hiddenLabel()
                    ->schema([
                        Grid::make(6)->childComponents([
                            TextEntry::make('product_name')
                                ->label('المنتج')
                                ->weight('bold'),

                            TextEntry::make('variant_name')
                                ->label('النسخة')
                                ->placeholder('-'),

                            TextEntry::make('sku')
                                ->label('SKU')
                                ->badge()
                                ->color('gray'),

                            TextEntry::make('quantity')
                                ->label('الكمية')
                                ->badge()
                                ->color('info'),

                            TextEntry::make('unit_price')
                                ->label('سعر الوحدة')
                                ->formatStateUsing(fn ($state) => number_format($state, 0, '', ',').' د.ع'),

                            TextEntry::make('line_total')
                                ->label('الإجمالي')
                                ->formatStateUsing(fn ($state) => number_format($state, 0, '', ',').' د.ع')
                                ->weight('bold')
                                ->color('success'),
                        ]),
                    ])
                    ->columns(1),
            ]);
    }

    /* ================== المجاميع ================== */

    protected static function sectionTotals(): Section
    {
        return Section::make('ملخص الطلب')
            ->icon('heroicon-o-calculator')
            ->columns(5)
            ->childComponents([
                TextEntry::make('subtotal')
                    ->label('المجموع الفرعي')
                    ->formatStateUsing(fn ($state) => number_format($state, 0, '', ',').' د.ع'),

                TextEntry::make('discount_total')
                    ->label('إجمالي الخصومات')
                    ->formatStateUsing(fn ($state) => $state > 0 ? '-'.number_format($state, 0, '', ',').' د.ع' : '-')
                    ->color('danger'),

                TextEntry::make('shipping_fee')
                    ->label('رسوم الشحن')
                    ->formatStateUsing(fn ($state) => number_format($state ?? 0, 0, '', ',').' د.ع'),

                TextEntry::make('cod_fee')
                    ->label('رسوم الدفع عند الاستلام')
                    ->formatStateUsing(fn ($state) => number_format($state ?? 0, 0, '', ',').' د.ع'),

                TextEntry::make('grand_total')
                    ->label('الإجمالي النهائي')
                    ->formatStateUsing(fn ($state) => number_format($state, 0, '', ',').' د.ع')
                    ->weight('bold')
                    ->size('lg')
                    ->color('success'),
            ]);
    }

    /* ================== الشحن ================== */

    protected static function sectionShipping(): Section
    {
        return Section::make('معلومات الشحن')
            ->icon('heroicon-o-truck')
            ->columns(3)
            ->collapsed()
            ->childComponents([
                TextEntry::make('shipping_method')
                    ->label('طريقة الشحن')
                    ->formatStateUsing(fn ($state) => $state?->label() ?? '-'),

                TextEntry::make('shippingCompany.name')
                    ->label('شركة الشحن')
                    ->placeholder('-'),

                TextEntry::make('shippingAddress.full_address')
                    ->label('عنوان الشحن')
                    ->placeholder('-'),
            ]);
    }

    /* ================== الملاحظات ================== */

    protected static function sectionNotes(): Section
    {
        return Section::make('ملاحظات')
            ->icon('heroicon-o-document-text')
            ->columns(2)
            ->collapsed()
            ->childComponents([
                TextEntry::make('customer_notes')
                    ->label('ملاحظات العميل')
                    ->placeholder('لا توجد ملاحظات'),

                TextEntry::make('internal_notes')
                    ->label('ملاحظات داخلية')
                    ->placeholder('لا توجد ملاحظات'),
            ]);
    }

    /* ================== التواريخ ================== */

    protected static function sectionTimestamps(): Section
    {
        return Section::make('التواريخ')
            ->icon('heroicon-o-clock')
            ->columns(4)
            ->collapsed()
            ->childComponents([
                TextEntry::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->dateTime('Y-m-d h:i A'),

                TextEntry::make('confirmed_at')
                    ->label('تاريخ التأكيد')
                    ->dateTime('Y-m-d h:i A')
                    ->placeholder('-'),

                TextEntry::make('completed_at')
                    ->label('تاريخ الإكمال')
                    ->dateTime('Y-m-d h:i A')
                    ->placeholder('-'),

                TextEntry::make('canceled_at')
                    ->label('تاريخ الإلغاء')
                    ->dateTime('Y-m-d h:i A')
                    ->placeholder('-'),
            ]);
    }
}
