<?php

namespace App\Filament\Resources\Orders\Schemas;

use App\Enums\OrderStatus;
use App\Enums\PaymentStatus;
use App\Enums\ShippingMethod;
use App\Models\Product;
use App\Models\ProductVariant;
use Filament\Forms\Components\Placeholder;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Components\Utilities\Set;
use Filament\Schemas\Schema;
use Illuminate\Support\HtmlString;

class OrderForm
{
    public static function configure(Schema $schema): Schema
    {
        return $schema->components([
            self::sectionBasicInfo(),
            self::sectionOrderItems(),
            self::sectionShipping(),
            self::sectionDiscounts(),
            self::sectionTotals(),
            self::sectionNotes(),
        ]);
    }

    /* ================== Helpers ================== */

    protected static function formatMoney(mixed $value): string
    {
        return number_format((int) $value, 0, '', ',');
    }

    protected static function toInt(mixed $value): int
    {
        if ($value === null || $value === '') {
            return 0;
        }

        $value = (string) $value;
        $clean = preg_replace('/[^\d]/u', '', $value);

        return (int) ($clean ?: 0);
    }

    protected static function moneyInput(string $name, string $label, bool $required = false): TextInput
    {
        $input = TextInput::make($name)
            ->label($label)
            ->inputMode('numeric')
            ->step(1)
            ->default('0')
            ->live(onBlur: true)
            ->afterStateHydrated(function ($state, callable $set) use ($name) {
                $set($name, self::formatMoney($state));
            })
            ->afterStateUpdated(function ($state, callable $set) use ($name) {
                $set($name, self::formatMoney($state));
            })
            ->dehydrateStateUsing(fn ($state) => self::toInt($state))
            ->rule('regex:/^[\d,]+$/u');

        return $required ? $input->required() : $input;
    }

    /* ================== المعلومات الأساسية ================== */

    protected static function sectionBasicInfo(): Section
    {
        return Section::make('معلومات الطلب')
            ->description('البيانات الأساسية للطلب')
            ->icon('heroicon-o-shopping-cart')
            ->collapsible()
            ->columnSpanFull()
            ->schema([
                Grid::make(3)->schema([
                    TextInput::make('order_number')
                        ->label('رقم الطلب')
                        ->disabled()
                        ->dehydrated(false)
                        ->placeholder('يُنشأ تلقائياً')
                        ->prefixIcon('heroicon-o-hashtag')
                        ->visibleOn('edit'),

                    Select::make('branch_id')
                        ->label('الفرع')
                        ->relationship('branch', 'name')
                        ->required()
                        ->searchable()
                        ->preload()
                        ->default(fn () => user_info('branch_id'))
                        ->prefixIcon('heroicon-o-building-storefront'),

                    Select::make('client_id')
                        ->label('العميل')
                        ->relationship('client', 'name')
                        ->required()
                        ->searchable()
                        ->preload()
                        ->createOptionForm([
                            TextInput::make('name')
                                ->label('اسم العميل')
                                ->required()
                                ->maxLength(255),
                            TextInput::make('phone')
                                ->label('رقم الهاتف')
                                ->tel()
                                ->maxLength(20),
                            TextInput::make('email')
                                ->label('البريد الإلكتروني')
                                ->email()
                                ->maxLength(255),
                        ])
                        ->prefixIcon('heroicon-o-user'),

                    Select::make('currency_id')
                        ->label('العملة')
                        ->relationship('currency', 'name')
                        ->required()
                        ->searchable()
                        ->preload()
                        ->default(1)
                        ->prefixIcon('heroicon-o-currency-dollar'),

                    TextInput::make('exchange_rate')
                        ->label('سعر الصرف')
                        ->numeric()
                        ->default(1)
                        ->required()
                        ->prefixIcon('heroicon-o-arrows-right-left'),

                    Select::make('status')
                        ->label('حالة الطلب')
                        ->options(OrderStatus::options())
                        ->default(OrderStatus::Pending->value)
                        ->required()
                        ->prefixIcon('heroicon-o-flag')
                        ->visibleOn('edit'),

                    Select::make('payment_status')
                        ->label('حالة الدفع')
                        ->options(collect(PaymentStatus::cases())->mapWithKeys(fn ($status) => [$status->value => $status->label()])->all())
                        ->default(PaymentStatus::UNPAID->value)
                        ->required()
                        ->prefixIcon('heroicon-o-credit-card')
                        ->visibleOn('edit'),
                ]),
            ]);
    }

    /* ================== عناصر الطلب ================== */

    protected static function sectionOrderItems(): Section
    {
        return Section::make('عناصر الطلب')
            ->description('المنتجات المطلوبة')
            ->icon('heroicon-o-cube')
            ->collapsible()
            ->columnSpanFull()
            ->schema([
                Repeater::make('items')
                    ->relationship()
                    ->label('المنتجات')
                    ->hiddenLabel()
                    ->collapsible()
                    ->collapsed(false)
                    ->defaultItems(1)
                    ->addActionLabel('+ إضافة منتج')
                    ->itemLabel(fn (array $state): ?string => $state['product_name'] ?? 'منتج جديد')
                    ->schema([
                        Grid::make(6)->schema([
                            Select::make('product_id')
                                ->label('المنتج')
                                ->options(fn () => Product::query()
                                    ->where('is_active', true)
                                    ->pluck('name', 'id'))
                                ->searchable()
                                ->required()
                                ->live()
                                ->afterStateUpdated(function ($state, Set $set, Get $get) {
                                    if (! $state) {
                                        return;
                                    }

                                    $product = Product::find($state);
                                    if ($product) {
                                        $set('product_name', $product->name);
                                        $set('product_variant_id', null);
                                        $set('variant_name', null);

                                        // إذا كان منتج بسيط، جلب السعر والتكلفة
                                        if (! $product->is_variable) {
                                            $variant = $product->getBaseVariant();
                                            if ($variant) {
                                                $set('product_variant_id', $variant->id);
                                                $set('sku', $variant->sku);
                                                $price = $variant->prices()->active()->first();
                                                if ($price) {
                                                    $set('unit_price', self::formatMoney($price->retail_price ?? 0));
                                                }
                                                $stock = $variant->stockBalances()->first();
                                                if ($stock) {
                                                    $set('unit_cost', self::formatMoney($stock->last_unit_cost ?? 0));
                                                }
                                            }
                                        }
                                    }

                                    self::calculateLineTotal($get, $set);
                                })
                                ->columnSpan(2),

                            Select::make('product_variant_id')
                                ->label('النسخة')
                                ->options(function (Get $get) {
                                    $productId = $get('product_id');
                                    if (! $productId) {
                                        return [];
                                    }

                                    $product = Product::find($productId);
                                    if (! $product || ! $product->is_variable) {
                                        return [];
                                    }

                                    return ProductVariant::query()
                                        ->where('product_id', $productId)
                                        ->where('is_active', true)
                                        ->get()
                                        ->mapWithKeys(fn ($v) => [$v->id => $v->variant_label ?: $v->sku])
                                        ->all();
                                })
                                ->searchable()
                                ->live()
                                ->visible(function (Get $get) {
                                    $productId = $get('product_id');
                                    if (! $productId) {
                                        return false;
                                    }

                                    $product = Product::find($productId);

                                    return $product?->is_variable ?? false;
                                })
                                ->afterStateUpdated(function ($state, Set $set, Get $get) {
                                    if (! $state) {
                                        return;
                                    }

                                    $variant = ProductVariant::find($state);
                                    if ($variant) {
                                        $set('variant_name', $variant->variant_label);
                                        $set('sku', $variant->sku);

                                        $price = $variant->prices()->active()->first();
                                        if ($price) {
                                            $set('unit_price', self::formatMoney($price->retail_price ?? 0));
                                        }

                                        $stock = $variant->stockBalances()->first();
                                        if ($stock) {
                                            $set('unit_cost', self::formatMoney($stock->last_unit_cost ?? 0));
                                        }
                                    }

                                    self::calculateLineTotal($get, $set);
                                })
                                ->columnSpan(1),

                            TextInput::make('quantity')
                                ->label('الكمية')
                                ->numeric()
                                ->default(1)
                                ->minValue(1)
                                ->required()
                                ->live(onBlur: true)
                                ->afterStateUpdated(fn (Get $get, Set $set) => self::calculateLineTotal($get, $set))
                                ->columnSpan(1),

                            self::moneyInput('unit_price', 'سعر الوحدة', true)
                                ->afterStateUpdated(fn (Get $get, Set $set) => self::calculateLineTotal($get, $set))
                                ->columnSpan(1),

                            self::moneyInput('discount_amount', 'الخصم')
                                ->afterStateUpdated(fn (Get $get, Set $set) => self::calculateLineTotal($get, $set))
                                ->columnSpan(1),

                            // Hidden fields
                            TextInput::make('product_name')->hidden()->dehydrated(true),
                            TextInput::make('variant_name')->hidden()->dehydrated(true),
                            TextInput::make('sku')->hidden()->dehydrated(true),
                            TextInput::make('unit_cost')->hidden()->dehydrated(true),
                            TextInput::make('line_total')->hidden()->dehydrated(true),
                        ]),

                        Placeholder::make('line_total_display')
                            ->label('إجمالي السطر')
                            ->content(function (Get $get): string {
                                $qty = (int) $get('quantity');
                                $price = self::toInt($get('unit_price'));
                                $discount = self::toInt($get('discount_amount'));
                                $total = ($qty * $price) - $discount;

                                return self::formatMoney($total).' د.ع';
                            }),
                    ])
                    ->live()
                    ->afterStateUpdated(fn (Get $get, Set $set) => self::calculateOrderTotals($get, $set)),
            ]);
    }

    /* ================== الشحن ================== */

    protected static function sectionShipping(): Section
    {
        return Section::make('معلومات الشحن')
            ->description('بيانات التوصيل والشحن')
            ->icon('heroicon-o-truck')
            ->collapsible()
            ->collapsed()
            ->columnSpanFull()
            ->schema([
                Grid::make(3)->schema([
                    Select::make('shipping_method')
                        ->label('طريقة الشحن')
                        ->options(ShippingMethod::options())
                        ->live()
                        ->prefixIcon('heroicon-o-truck'),

                    Select::make('shipping_company_id')
                        ->label('شركة الشحن')
                        ->relationship('shippingCompany', 'name')
                        ->searchable()
                        ->preload()
                        ->visible(fn (Get $get) => $get('shipping_method') === ShippingMethod::Company->value)
                        ->prefixIcon('heroicon-o-building-office'),

                    Select::make('shipping_address_id')
                        ->label('عنوان الشحن')
                        ->relationship('shippingAddress', 'address')
                        ->searchable()
                        ->preload()
                        ->prefixIcon('heroicon-o-map-pin'),

                    self::moneyInput('shipping_fee', 'رسوم الشحن')
                        ->live(onBlur: true)
                        ->afterStateUpdated(fn (Get $get, Set $set) => self::calculateOrderTotals($get, $set)),

                    self::moneyInput('cod_fee', 'رسوم الدفع عند الاستلام')
                        ->live(onBlur: true)
                        ->afterStateUpdated(fn (Get $get, Set $set) => self::calculateOrderTotals($get, $set)),
                ]),
            ]);
    }

    /* ================== الخصومات ================== */

    protected static function sectionDiscounts(): Section
    {
        return Section::make('الخصومات والعروض')
            ->description('الخصومات والكوبونات المطبقة')
            ->icon('heroicon-o-receipt-percent')
            ->collapsible()
            ->collapsed()
            ->columnSpanFull()
            ->schema([
                Grid::make(3)->schema([
                    Select::make('promotion_id')
                        ->label('العرض')
                        ->relationship('promotion', 'name')
                        ->searchable()
                        ->preload()
                        ->prefixIcon('heroicon-o-gift'),

                    Select::make('coupon_id')
                        ->label('كوبون الخصم')
                        ->relationship('coupon', 'code')
                        ->searchable()
                        ->preload()
                        ->prefixIcon('heroicon-o-ticket'),

                    self::moneyInput('client_discount_amount', 'خصم العميل المسجل'),
                ]),
            ]);
    }

    /* ================== المجاميع ================== */

    protected static function sectionTotals(): Section
    {
        return Section::make('ملخص الطلب')
            ->description('إجماليات الطلب')
            ->icon('heroicon-o-calculator')
            ->collapsible()
            ->columnSpanFull()
            ->schema([
                Grid::make(4)->schema([
                    self::moneyInput('subtotal', 'المجموع الفرعي')
                        ->disabled()
                        ->dehydrated(true),

                    self::moneyInput('discount_total', 'إجمالي الخصومات')
                        ->disabled()
                        ->dehydrated(true),

                    Placeholder::make('fees_display')
                        ->label('الرسوم الإضافية')
                        ->content(function (Get $get): string {
                            $shipping = self::toInt($get('shipping_fee'));
                            $cod = self::toInt($get('cod_fee'));

                            return self::formatMoney($shipping + $cod).' د.ع';
                        }),

                    Placeholder::make('grand_total_display')
                        ->label('الإجمالي النهائي')
                        ->content(function (Get $get): HtmlString {
                            $subtotal = self::toInt($get('subtotal'));
                            $discount = self::toInt($get('discount_total'));
                            $shipping = self::toInt($get('shipping_fee'));
                            $cod = self::toInt($get('cod_fee'));
                            $total = $subtotal - $discount + $shipping + $cod;

                            return new HtmlString('<span class="text-xl font-bold text-primary-600">'.self::formatMoney($total).' د.ع</span>');
                        }),

                    TextInput::make('grand_total')
                        ->hidden()
                        ->dehydrated(true),
                ]),
            ]);
    }

    /* ================== الملاحظات ================== */

    protected static function sectionNotes(): Section
    {
        return Section::make('ملاحظات')
            ->description('ملاحظات العميل والملاحظات الداخلية')
            ->icon('heroicon-o-document-text')
            ->collapsible()
            ->collapsed()
            ->columnSpanFull()
            ->schema([
                Grid::make(2)->schema([
                    Textarea::make('customer_notes')
                        ->label('ملاحظات العميل')
                        ->rows(3)
                        ->placeholder('ملاحظات من العميل...'),

                    Textarea::make('internal_notes')
                        ->label('ملاحظات داخلية')
                        ->rows(3)
                        ->placeholder('ملاحظات للموظفين فقط...'),
                ]),
            ]);
    }

    /* ================== حسابات ================== */

    protected static function calculateLineTotal(Get $get, Set $set): void
    {
        $qty = (int) $get('quantity');
        $price = self::toInt($get('unit_price'));
        $discount = self::toInt($get('discount_amount'));
        $total = ($qty * $price) - $discount;
        $set('line_total', $total);
    }

    protected static function calculateOrderTotals(Get $get, Set $set): void
    {
        $items = $get('items') ?? [];

        $subtotal = 0;
        $discountTotal = 0;

        foreach ($items as $item) {
            $qty = (int) ($item['quantity'] ?? 0);
            $price = self::toInt($item['unit_price'] ?? 0);
            $discount = self::toInt($item['discount_amount'] ?? 0);

            $subtotal += ($qty * $price);
            $discountTotal += $discount;
        }

        $clientDiscount = self::toInt($get('client_discount_amount'));
        $discountTotal += $clientDiscount;

        $shipping = self::toInt($get('shipping_fee'));
        $cod = self::toInt($get('cod_fee'));

        $grandTotal = $subtotal - $discountTotal + $shipping + $cod;

        $set('subtotal', self::formatMoney($subtotal));
        $set('discount_total', self::formatMoney($discountTotal));
        $set('grand_total', $grandTotal);
    }
}
