<?php

namespace App\Filament\Resources\Orders\Pages;

use App\Enums\OrderStatus;
use App\Filament\Resources\Orders\OrderResource;
use App\Services\Orders\OrderInvoiceService;
use App\Services\Orders\OrderService;
use Filament\Actions\Action;
use Filament\Actions\ActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Actions\ViewAction;
use Filament\Forms\Components\Textarea;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\EditRecord;

class EditOrder extends EditRecord
{
    protected static string $resource = OrderResource::class;

    protected function getHeaderActions(): array
    {
        $orderService = app(OrderService::class);
        $transitions = $orderService->getAvailableTransitions($this->record);

        $statusActions = [];
        foreach ($transitions as $transition) {
            $statusActions[] = Action::make('transition_'.$transition['status']->value)
                ->label($transition['label'])
                ->icon($transition['status']->icon())
                ->color($transition['color'])
                ->requiresConfirmation()
                ->modalHeading($transition['label'])
                ->modalDescription('هل أنت متأكد من '.$transition['label'].'؟')
                ->form(function () use ($transition) {
                    if (in_array($transition['status'], [OrderStatus::Canceled, OrderStatus::Refunded])) {
                        return [
                            Textarea::make('reason')
                                ->label('السبب')
                                ->placeholder('أدخل سبب '.$transition['label'].'...')
                                ->rows(2),
                        ];
                    }

                    return [];
                })
                ->action(function (array $data) use ($orderService, $transition) {
                    $reason = $data['reason'] ?? null;
                    $success = $orderService->transition($this->record, $transition['status'], $reason);

                    if ($success) {
                        Notification::make()
                            ->title('تم '.$transition['label'].' بنجاح')
                            ->success()
                            ->send();

                        $this->refreshFormData(['status', 'payment_status']);
                    } else {
                        Notification::make()
                            ->title('فشل '.$transition['label'])
                            ->danger()
                            ->send();
                    }
                });
        }

        return [
            ViewAction::make(),

            ActionGroup::make($statusActions)
                ->label('تغيير الحالة')
                ->icon('heroicon-o-arrow-path')
                ->button()
                ->color('primary')
                ->visible(fn () => count($statusActions) > 0),

            Action::make('create_invoice')
                ->label('إنشاء فاتورة')
                ->icon('heroicon-o-document-text')
                ->color('info')
                ->visible(fn () => ! in_array($this->record->status, [OrderStatus::Pending, OrderStatus::Canceled]))
                ->action(function () {
                    try {
                        $invoiceService = app(OrderInvoiceService::class);
                        $invoice = $invoiceService->createInvoiceFromOrder($this->record);

                        Notification::make()
                            ->title('تم إنشاء الفاتورة بنجاح')
                            ->body('رقم الفاتورة: '.$invoice->invoice_number)
                            ->success()
                            ->send();
                    } catch (\Exception $e) {
                        Notification::make()
                            ->title('خطأ')
                            ->body($e->getMessage())
                            ->danger()
                            ->send();
                    }
                }),

            Action::make('print')
                ->label('طباعة')
                ->icon('heroicon-o-printer')
                ->color('gray')
                ->url(fn () => route('orders.print', $this->record), shouldOpenInNewTab: true),

            DeleteAction::make()
                ->visible(fn () => $this->record->status === OrderStatus::Pending),
        ];
    }

    protected function getRedirectUrl(): string
    {
        return $this->getResource()::getUrl('index');
    }
}
