<?php

namespace App\Filament\Resources\Inventory\StockMovements\Tables;

use App\Enums\MovementDirection;
use Filament\Actions\BulkActionGroup;
use Filament\Forms\Components\DatePicker;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class StockMovementsTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->defaultSort('created_at', 'desc')
            ->columns([
                TextColumn::make('variant.product.name')
                    ->label('المنتج')
                    ->searchable(),
                TextColumn::make('variant.variant_label')
                    ->label('النسخ')
                    ->tooltip(fn ($record) => $record->variant?->sku) // اختياري
                    ->searchable(query: function ($query, string $term) {
                        // ابحث في SKU وقيم الخصائص
                        $query->orWhereHas(
                            'variant',
                            fn ($q) => $q->where('sku', 'like', "%{$term}%")
                        )->orWhereHas(
                            'variant.attributeValues',
                            fn ($q) => $q->where('display_value', 'like', "%{$term}%")
                        );
                    }),
                TextColumn::make('direction')
                    ->label('نوع العملية')
                    ->formatStateUsing(function ($state) {
                        // $state قد يكون string أو MovementDirection حسب التحميل
                        $enum = $state instanceof MovementDirection ? $state : MovementDirection::from($state);

                        return $enum->label();
                    })
                    ->badge()
                    ->color(function ($state) {
                        $enum = $state instanceof MovementDirection ? $state : MovementDirection::from($state);

                        return match ($enum) {
                            MovementDirection::IN => 'success',
                            MovementDirection::OUT => 'danger',
                            MovementDirection::ADJ => 'warning',
                        };
                    }),
                TextColumn::make('quantity')
                    ->label('الكمية')
                    ->numeric(decimalPlaces: 0, thousandsSeparator: ',', locale: 'en')
                    ->weight('bold')
                    ->sortable(),
                TextColumn::make('unit_cost')
                    ->label('تكلفة/وحدة')
                    ->formatStateUsing(function ($state) {
                        return number_format($state);
                    })
                    ->weight('bold')
                    ->suffix(' د.ع ')
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('warehouse.name')
                    ->label('المستودع')
                    ->sortable()
                    ->searchable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('moved_at')
                    ->label('التاريخ')
                    ->dateTime('Y-m-d')
                    ->sortable(),
                TextColumn::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->dateTime('Y-m-d h:i A')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('updated_at')
                    ->label('آخر تحديث')
                    ->dateTime('Y-m-d h:i A')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                SelectFilter::make('warehouse.city_id')
                    ->label('المدينة')
                    ->preload()
                    ->relationship('warehouse.city', 'name')
                    ->searchable(),
                SelectFilter::make('warehouse_id')
                    ->label('المستودع')
                    ->relationship('warehouse', 'name')
                    ->preload()
                    ->searchable(),
                SelectFilter::make('direction')
                    ->label('العملية')
                    ->options([
                        MovementDirection::IN->value => MovementDirection::IN->label(),
                        MovementDirection::OUT->value => MovementDirection::OUT->label(),
                        MovementDirection::ADJ->value => MovementDirection::ADJ->label(),
                    ])
                    ->searchable(),
                Filter::make('period')
                    ->label('فترة')
                    ->schema([
                        DatePicker::make('from')->label('من'),
                        DatePicker::make('to')->label('إلى'),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when($data['from'] ?? null, fn ($q, $date) => $q->whereDate('moved_at', '>=', $date))
                            ->when($data['to'] ?? null, fn ($q, $date) => $q->whereDate('moved_at', '<=', $date));
                    }),

            ])
            ->recordActions([])
            ->toolbarActions([
                BulkActionGroup::make([]),
            ]);
    }
}
