<?php

namespace App\Filament\Resources\Inventory\StockMovements\Schemas;

use App\Models\Product;
use Filament\Actions\Action;
use Filament\Schemas\Schema;
use App\Models\ProductVariant;
use App\Models\BranchWarehouse;
use Illuminate\Support\HtmlString;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Toggle;
use Filament\Schemas\Components\Grid;
use Illuminate\Support\Facades\Cache;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Notifications\Notification;
use Filament\Schemas\Components\Section;
use Filament\Infolists\Components\TextEntry;
use Coolsam\Flatpickr\Forms\Components\Flatpickr;

class StockMovementForm
{
    /** Cache duration in seconds */
    protected const CACHE_DURATION = 3600; // 1 hour

    /** Maximum search results */
    protected const MAX_SEARCH_RESULTS = 20;

    /** Default preload limit */
    protected const PRELOAD_LIMIT = 30;

    public static function configure(Schema $schema): Schema
    {
        return $schema->components([
            // ═══════════════════════════════════════════════════════════════
            // 🎯 القسم الأول: معلومات الحركة الأساسية
            // ═══════════════════════════════════════════════════════════════
            Section::make('📋 معلومات الحركة')
                ->description('البيانات الأساسية لحركة المخزون')
                ->icon('heroicon-o-clipboard-document-check')
                ->iconColor('primary')
                ->extraAttributes([
                    'class' => 'overflow-hidden rounded-2xl border border-blue-100 dark:border-blue-900 bg-gradient-to-br from-blue-50/70 via-indigo-50/60 to-slate-50/70 dark:from-slate-900/70 dark:via-slate-900 dark:to-slate-950 shadow-sm hover:shadow-lg transition-all duration-300',
                ])
                ->schema([
                    Grid::make(12)
                        ->schema([
                            Select::make('warehouse_id')
                                ->label('🏢 المستودع')
                                ->relationship(
                                    'warehouse',
                                    'name',
                                    fn($query) => user_info()?->admin()
                                        ? null
                                        : $query->whereIn(
                                            'id',
                                            Cache::remember(
                                                'branch_warehouses_' . user_info('branch_id'),
                                                self::CACHE_DURATION,
                                                fn() => BranchWarehouse::where('branch_id', user_info('branch_id'))
                                                    ->pluck('warehouse_id')
                                            )
                                        )
                                )
                                ->required()
                                ->searchable()
                                ->preload()
                                ->native(false)
                                ->prefixIcon('heroicon-o-building-storefront')
                                ->helperText('اختر المستودع الذي سيتم إدخال البضاعة إليه.')
                                ->extraAttributes(['class' => 'font-semibold'])
                                ->columnSpan(4),

                            Flatpickr::make('moved_at')
                                ->label('📅 تاريخ ووقت الإدخال')
                                ->default(now())
                                ->required()
                                ->time(true)
                                ->time24hr(false)
                                ->seconds(false)
                                ->format('Y-m-d H:i')
                                ->altInput(true)
                                ->altFormat('Y-m-d h:i K')
                                ->locale('ar')
                                ->prefixIcon('heroicon-o-calendar-days')
                                ->helperText('يمكنك تعديل التاريخ والوقت حسب الحاجة.')
                                ->extraAttributes(['class' => 'font-semibold'])
                                ->columnSpan(4),

                            TextInput::make('reference_number')
                                ->label('🔖 رقم المرجع')
                                ->maxLength(100)
                                ->placeholder('مثال: INV-2024-001')
                                ->prefixIcon('heroicon-o-hashtag')
                                ->helperText('رقم الفاتورة أو المرجع الخارجي (اختياري).')
                                ->extraAttributes(['class' => 'font-semibold'])
                                ->columnSpan(4),
                        ]),
                ])
                ->collapsible()
                ->collapsed(false)
                ->columnSpanFull(),

            // ═══════════════════════════════════════════════════════════════
            // 📦 القسم الثاني: عناصر الحركة
            // ═══════════════════════════════════════════════════════════════

            // ═══════════════════════════════════════════════════════════════
            // 📝 ملاحظات عامة
            // ═══════════════════════════════════════════════════════════════
            Section::make('🗒️ ملاحظات إضافية')
                ->description('ملاحظات على مستوى الحركة بالكامل')
                ->icon('heroicon-o-document-text')
                ->iconColor('success')
                ->extraAttributes([
                    'class' => 'overflow-hidden rounded-2xl border border-emerald-100 dark:border-emerald-900 bg-gradient-to-br from-emerald-50/60 to-green-50/40 dark:from-slate-950 dark:via-slate-950 dark:to-slate-950/90 shadow-sm hover:shadow-lg transition-all duration-300',
                ])
                ->schema([
                    Textarea::make('general_notes')
                        ->label('📄 ملاحظات عامة')
                        ->rows(3)
                        ->placeholder('أي تفاصيل إضافية تتعلق بهذه الحركة...')
                        ->extraAttributes([
                            'class' => 'rounded-xl bg-white dark:bg-slate-950 border border-emerald-100 dark:border-emerald-800 text-[13px]',
                        ])
                        ->columnSpanFull(),
                ])
                ->collapsible()
                ->collapsed(true)
                ->columnSpanFull(),
        ]);
    }

    // ═══════════════════════════════════════════════════════════════
    // ⚙️ توابع مساعدة
    // ═══════════════════════════════════════════════════════════════

    protected static function getCachedProductName(int $productId): string
    {
        $cacheKey = 'product_name_' . $productId;

        return Cache::remember($cacheKey, self::CACHE_DURATION, function () use ($productId) {
            return Product::query()->whereKey($productId)->value('name') ?? 'منتج';
        });
    }

    protected static function normalizeArabic(string $text): string
    {
        static $replacements = [
            'إ' => 'ا',
            'أ' => 'ا',
            'آ' => 'ا',
            'ى' => 'ي',
            'ي' => 'ي',
            'ة' => 'ه',
        ];

        $text = trim($text);
        $text = preg_replace('/\s+/', ' ', $text);
        $text = preg_replace('/^ال\s+/', '', $text);
        $text = str_replace(array_keys($replacements), array_values($replacements), $text);

        return mb_strtolower($text);
    }

    protected static function tokenize(string $text): array
    {
        $normalized = static::normalizeArabic($text);
        $parts      = preg_split('/\s+/u', $normalized, -1, PREG_SPLIT_NO_EMPTY);

        return array_values(array_filter($parts, fn($w) => mb_strlen($w) > 1));
    }
}
