<?php

namespace App\Filament\Resources\Inventory\StockMovements\Pages;

use App\Enums\MovementDirection;
use App\Filament\Resources\Inventory\StockMovements\StockMovementResource;
use App\Models\Product;
use App\Models\ProductVariant;
use App\Models\StockBalance;
use App\Models\StockMovement;
use Filament\Resources\Pages\CreateRecord;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

class CreateStockMovement extends CreateRecord
{
    protected static string $resource = StockMovementResource::class;

    /**
     * نستولي على عملية الإنشاء بالكامل: ننشئ حركات متعددة (IN) ونحدّث الأرصدة.
     */
    protected function handleRecordCreation(array $data): Model
    {
        return DB::transaction(function () use ($data) {
            // 0) تحقّق من السطور
            $lines = collect($data['variant_lines'] ?? [])->filter(function ($line) {
                return (int)($line['quantity'] ?? 0) > 0 && (
                    !empty($line['product_variant_id']) || !empty($line['pending_new_variant'])
                );
            })->values();

            if ($lines->isEmpty()) {
                throw new \RuntimeException('لم يتم إدخال أي نسخة/كمية صالحة.');
            }

            // 1) المنتج: موجود أو جديد
            $productId = $data['product_id'] ?? null;

            if (($data['is_new_product'] ?? false) && empty($productId)) {
                // يمكنك استدعاء خدمتك الخاصة إن لديك منطق أوسع:
                // $product = app(\App\Services\ProductCreationService::class)->create([...]);

                $product = Product::create([
                    'name' => (string) ($data['product_name_display'] ?? 'منتج بدون اسم'),
                    'is_active' => true,
                ]);

                $productId = $product->id;
            } else {
                $product = $productId ? Product::findOrFail($productId) : null;
            }

            // 2) جهّز القيم المشتركة بين كل الحركات
            $warehouseId      = (int) $data['warehouse_id'];
            $movedAt          = $data['moved_at'] ?? now();
            $referenceNumber  = $data['reference_number'] ?? null;
            $generalNotes     = $data['general_notes'] ?? null;

            // 3) مرّ على كل سطر Variant
            $firstMovement = null;

            foreach ($lines as $line) {
                // 3.a) احصل/أنشئ الـ Variant
                $variantId = $line['product_variant_id'] ?? null;

                if (!$variantId && !empty($line['pending_new_variant'])) {
                    if (!$productId) {
                        throw new \RuntimeException('لا يمكن إنشاء نسخة جديدة قبل إنشاء/تحديد المنتج.');
                    }
                    $pending = $line['pending_new_variant'] ?? [];
                    $sku     = $pending['sku'] ?: null;

                    $variant = ProductVariant::create([
                        'product_id' => $productId,
                        'sku'        => $sku, // سيتولّد تلقائيًا إن null حسب booted()
                        'cost_price' => (int)($pending['cost_price'] ?? 0),
                        'price'      => 0,
                        'stock_quantity'  => 0,
                        'min_stock_level' => 0,
                        'is_active'       => true,
                        'is_default'      => false,
                    ]);

                    $variantId = $variant->id;
                }

                $variant = ProductVariant::findOrFail($variantId);

                // 3.b) تحديث سعر التكلفة إن طُلِب (اختياري)
                if (!empty($line['override_cost']) && isset($line['new_cost_price'])) {
                    $variant->cost_price = (int) $line['new_cost_price'];
                    $variant->save();
                }

                $qty = (int)($line['quantity'] ?? 0);
                if ($qty <= 0) continue;

                // 3.c) إنشاء حركة IN
                $movement = StockMovement::create([
                    'warehouse_id'       => $warehouseId,
                    'product_variant_id' => $variant->id,
                    'direction'          => MovementDirection::IN, // enum cast مُفترض
                    'quantity'           => $qty,
                    'unit_cost'          => $variant->cost_price ?? 0, // عدّل حسب أعمدتك
                    'reference_number'   => $referenceNumber,
                    'moved_at'           => $movedAt,
                    'memo'               => $line['memo'] ?? null,
                    'general_notes'      => $generalNotes, // إن كان لديك عمود عام للحركة
                ]);

                // 3.d) تحديث الرصيد
                $balance = StockBalance::firstOrCreate([
                    'warehouse_id' => $warehouseId,
                    'product_variant_id' => $variant->id,
                ]);

                // اتجاه IN → +qty (استخدم enum->sign() لو أردت)
                $balance->quantity = (int) ($balance->quantity ?? 0) + $qty;
                $balance->save();

                $firstMovement ??= $movement;
            }

            // يجب إعادة Model واحد ليتصالح مع Filament
            return $firstMovement;
        });
    }
}
