<?php

namespace App\Filament\Resources\Inventory\StockBalances\Tables;

use App\Models\Barcode;
use App\Models\Warehouse;
use App\Models\StockBalance;
use Filament\Tables\Table;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;
use Filament\Support\Enums\FontWeight;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Illuminate\Database\Eloquent\Builder;

class StockBalancesTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->modifyQueryUsing(function (Builder $q) {
                $q
                    ->where('branch_id', user_info('branch_id'))
                    ->with([
                        'warehouse.branches',
                        'variant',
                        'variant.product',
                        'variant.barcodes' => fn($qq) => $qq
                            ->where('is_primary', true)
                            ->where('is_active', true),
                        'variant.attributeValues.attribute',
                    ]);
            })
            ->columns([
                // المستودع
                TextColumn::make('warehouse.name')
                    ->label('المستودع')
                    ->badge()
                    ->color('danger')
                    ->sortable()
                    ->searchable(),

                TextColumn::make('branch_name')
                    ->label('الفرع')
                    ->badge()
                    ->sortable()
                    ->searchable()
                    ->getStateUsing(function ($record) {
                        // تأكيد وجود المستودع لتفادي الخطأ:
                        if (! $record->warehouse) {
                            return '-';
                        }

                        // جلب الفروع المرتبطة بالمستودع
                        $branches = $record->warehouse->branches;

                        if ($branches->isEmpty()) {
                            return '-';
                        }

                        // لو عندك فرع أساسي في pivot:
                        $primary = $branches->firstWhere('pivot.is_primary', true);

                        if ($primary) {
                            return $primary->name;
                        }

                        // أو نرجع أول فرع
                        return $branches->first()->name;
                    }),

                // المنتج
                TextColumn::make('variant.product.name')
                    ->label('المنتج')
                    ->wrap()
                    ->limit(40)
                    ->sortable()
                    ->searchable()
                    ->weight(FontWeight::Bold)
                    ->color('info'),

                // عنوان المتغيّر من الخصائص
                TextColumn::make('variant_title')
                    ->label('النسخ')
                    ->state(function (StockBalance $record) {
                        $pv = $record->variant;
                        if (! $pv) {
                            return '—';
                        }

                        $vals = $pv->attributeValues ?? collect();

                        $pairs = $vals->map(function ($v) {
                            $a = $v->attribute->display_name ?? null;
                            $d = $v->display_value ?? null;

                            return ($a && $d) ? "{$a} {$d}" : null;
                        })->filter()->values();

                        return $pairs->isNotEmpty() ? $pairs->join(' - ') : '—';
                    })
                    ->wrap()
                    ->weight(FontWeight::Bold)
                    ->toggleable(isToggledHiddenByDefault: true),

                // SKU
                TextColumn::make('variant.sku')
                    ->label('SKU')
                    ->badge()
                    ->copyable()
                    ->copyMessage('نُسخ SKU')
                    ->sortable()
                    ->searchable()
                    ->toggleable(isToggledHiddenByDefault: true),

                // الباركود الأساسي
                TextColumn::make('primary_barcode')
                    ->label('الباركود')
                    ->weight(FontWeight::Bold)
                    ->state(function (StockBalance $r) {
                        // أولوية: accessor من المتغيّر
                        $code = $r->variant?->primary_barcode;

                        // احتياط: استعلام مباشر إن لزم
                        if (! $code) {
                            $code = Barcode::query()
                                ->where('product_variant_id', $r->product_variant_id)
                                ->where('is_primary', true)
                                ->where('is_active', true)
                                ->value('code');
                        }

                        return $code ?: '—';
                    })
                    ->html()
                    ->copyable()
                    ->copyableState(function (StockBalance $r) {
                        return $r->variant?->primary_barcode
                            ?? Barcode::where('product_variant_id', $r->product_variant_id)
                            ->where('is_primary', true)
                            ->where('is_active', true)
                            ->value('code');
                    })
                    ->copyMessage('نُسخ الباركود')
                    ->searchable(query: function (Builder $query, string $search) {
                        $query->whereHas('variant.barcodes', function (Builder $q) use ($search) {
                            $q->where('is_active', true)
                                ->where('code', 'like', "%{$search}%");
                        });
                    })
                    ->toggleable(isToggledHiddenByDefault: true),

                // الرصيد الفعلي في المستودع
                TextColumn::make('on_hand')
                    ->label('الرصيد الفعلي')
                    ->numeric(decimalPlaces: 0, thousandsSeparator: ',', locale: 'en')
                    ->badge()
                    ->colors([
                        'danger'  => fn($state, StockBalance $r) => $r->is_low,
                        'warning' => fn($state, StockBalance $r) => ! $r->is_low && $state <= 5,
                        'success' => fn() => true,
                    ])
                    ->sortable(),

                // المحجوز للطلبات (اختياري)
                TextColumn::make('reserved_qty')
                    ->label('محجوز للطلبات')
                    ->numeric(decimalPlaces: 0, thousandsSeparator: ',', locale: 'en')
                    ->sortable()
                    ->badge()
                    ->colors([
                        'danger'  => fn($state, StockBalance $r) => $r->is_low,
                        'warning' => fn($state, StockBalance $r) => ! $r->is_low && $state <= 5,
                        'success' => fn() => true,
                    ])
                    ->toggleable(isToggledHiddenByDefault: true),

                // المتوفر = on_hand - reserved_qty (من الـ accessor)
                TextColumn::make('available')
                    ->label('المتوفر')
                    ->numeric(decimalPlaces: 0, thousandsSeparator: ',', locale: 'en')
                    ->state(fn(StockBalance $r) => $r->available)
                    ->sortable()
                    ->badge()
                    ->colors([
                        'danger'  => fn($state, StockBalance $r) => $r->is_low,
                        'warning' => fn($state, StockBalance $r) => ! $r->is_low && $state <= 5,
                        'success' => fn() => true,
                    ])
                    ->toggleable(isToggledHiddenByDefault: true),

                // الحد الأدنى وحد إعادة الطلب... إلخ (مخفية افتراضياً)
                TextColumn::make('min_stock_level')
                    ->label('الحد الأدنى')
                    ->numeric(decimalPlaces: 0, thousandsSeparator: ',', locale: 'en')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('reorder_point')
                    ->label('نقطة إعادة الطلب')
                    ->numeric(decimalPlaces: 0, thousandsSeparator: ',', locale: 'en')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('reorder_quantity')
                    ->label('كمية إعادة الطلب')
                    ->numeric(decimalPlaces: 0, thousandsSeparator: ',', locale: 'en')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                // آخر كلفة
                TextColumn::make('last_unit_cost')
                    ->label('آخر كلفة')
                    ->formatStateUsing(fn($state) => number_format($state))
                    ->suffix(' د.ع')
                    ->sortable()
                    ->weight(FontWeight::Bold)
                    ->toggleable(isToggledHiddenByDefault: true),

                // آخر تحديث
                TextColumn::make('updated_at')
                    ->label('آخر تحديث')
                    ->dateTime('Y-m-d H:i A')
                    ->sortable()
                    ->weight(FontWeight::Bold)
                    ->toggleable(isToggledHiddenByDefault: true),
            ])

            ->filters([
                // ✅ فلتر واحد فقط - المستودعات المرتبطة بالفرع الحالي
                SelectFilter::make('warehouse_id')
                    ->label('المستودع')
                    ->options(function () {
                        $branchId = user_info('branch_id');

                        return Warehouse::query()
                            ->where('is_active', true)  // المستودعات النشطة فقط
                            ->forBranch($branchId)      // المستودعات المرتبطة بالفرع
                            ->orderBy('name')
                            ->pluck('name', 'id')
                            ->all();
                    })
                    ->attribute('warehouse_id'),
            ])

            ->searchPlaceholder('ابحث بالمنتج أو SKU أو الباركود…')
            ->defaultSort('updated_at', 'desc')
            ->paginated([25, 50, 100])
            ->recordActions([
                // EditAction::make()->label('تعديل'),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make()->label('حذف المحدد'),
                ]),
            ]);
    }
}
