<?php

namespace App\Filament\Resources\InstallmentSchedules\Tables;

use Carbon\Carbon;
use Filament\Tables\Table;
use Filament\Actions\EditAction;
use App\Models\InstallmentSchedule;
use Filament\Tables\Filters\Filter;
use Filament\Actions\BulkActionGroup;
use Filament\Forms\Components\Select;
use Filament\Actions\DeleteBulkAction;
use Filament\Tables\Columns\TextColumn;
use Filament\Forms\Components\DatePicker;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Filters\TernaryFilter;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Components\Utilities\Set;

class InstallmentSchedulesTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->modifyQueryUsing(
                fn(Builder $query) =>
                $query->whereIn('status', ['pending', 'overdue'])
            )
            ->columns([
                TextColumn::make('plan.client.name')
                    ->label('العميل')
                    ->searchable()
                    ->sortable(),

                TextColumn::make('plan.invoice_type')
                    ->label('نوع الفاتورة')
                    ->formatStateUsing(function ($state, $record) {
                        $type = class_basename($record->plan->invoice_type);
                        $id   = $record->plan->invoice_id;

                        // تحويل الاسم إلى عربي
                        $arabicType = match ($type) {
                            'Sale'     => 'فاتورة بيع',
                            'Purchase' => 'فاتورة شراء',
                            default    => 'فاتورة',
                        };

                        return "{$arabicType} - {$id} # ";
                    })
                    ->url(function ($record) {
                        $type = class_basename($record->plan->invoice_type);
                        $id   = $record->plan->invoice_id;

                        // بناء الرابط حسب النوع
                        return match ($type) {
                            'Sale'     => route('filament.admin.resources.sales.edit', $id),
                            'Purchase' => route('filament.admin.resources.purchases.edit', $id),
                            default    => null,
                        };
                    }, shouldOpenInNewTab: true)
                    ->icon('phosphor-link')
                    ->color('primary')
                    ->sortable()
                    ->searchable(),

                TextColumn::make('sequence')
                    ->label('رقم القسط')
                    ->sortable(),

                TextColumn::make('amount')
                    ->label('المبلغ')
                    ->formatStateUsing(fn($state) => number_format((int) $state))
                    ->suffix(' دينار')
                    ->sortable(),
                TextColumn::make('status')
                    ->label('الحالة')
                    ->badge()
                    ->colors([
                        'warning' => 'pending',
                        'danger'  => 'overdue',
                        'success' => 'paid',
                        'gray'    => 'canceled',
                    ])
                    ->formatStateUsing(fn(string $state) => match ($state) {
                        'pending'  => 'معلّق',
                        'overdue'  => 'متأخر',
                        'paid'     => 'مدفوع',
                        'canceled' => 'ملغى',
                        default    => $state,
                    }),

                TextColumn::make('due_date')
                    ->label('تاريخ الاستحقاق')
                    ->date('Y-m-d')
                    ->sortable(),

                TextColumn::make('days_overdue')
                    ->label('أيام التأخر')
                    ->state(function (InstallmentSchedule $record) {
                        if ($record->status !== 'overdue') return '-';
                        return Carbon::parse($record->due_date)->diffInDays(Carbon::today());
                    })
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('plan.user.name')
                    ->label('الموظف')
                    ->badge()
                    ->sortable(),

                TextColumn::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->dateTime('Y-m-d H:i:s')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

            ])
            ->filters([
                Filter::make('due_range')
                    ->label('مدى تاريخ الاستحقاق')
                    ->schema([
                        Select::make('range')
                            ->label('المدى')
                            ->options([
                                'today'  => 'اليوم',
                                'week'   => 'هذا الأسبوع',
                                'month'  => 'هذا الشهر',
                                'custom' => 'مخصص',
                            ])
                            ->native(false)
                            ->live() // ← مهم: يجعل التغيير يعيد رسم الحقول فورًا
                            ->afterStateUpdated(function (Set $set, $state) {
                                // نظّف التواريخ عندما لا يكون "مخصص"
                                if ($state !== 'custom') {
                                    $set('from', null);
                                    $set('to', null);
                                }
                            })
                            ->required(),

                        DatePicker::make('from')
                            ->label('من تاريخ')
                            ->native(false)
                            ->closeOnDateSelection()
                            ->visible(fn(Get $get) => $get('range') === 'custom')
                            ->requiredIf('range', 'custom'),

                        DatePicker::make('to')
                            ->label('إلى تاريخ')
                            ->native(false)
                            ->closeOnDateSelection()
                            ->visible(fn(Get $get) => $get('range') === 'custom')
                            ->requiredIf('range', 'custom'),
                    ]),

                TernaryFilter::make('only_overdue')
                    ->label('فقط المتأخرة')
                    ->placeholder('الكل')
                    ->trueLabel('نعم')
                    ->falseLabel('إخفاء المتأخرة')
                    ->queries(
                        true: fn(Builder $q) => $q->where('status', 'overdue'),
                        false: fn(Builder $q) => $q->where('status', '!=', 'overdue'),
                        blank: fn(Builder $q) => $q,
                    ),
            ])
            ->recordActions([])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ]);
    }
}
