<?php

namespace App\Filament\Resources\Finance\Settleds\Tables;

use App\Filament\Resources\Clients\ClientResource;
use App\Models\Client;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Enums\FiltersLayout;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class SettledsTable
{
    public static function configure(Table $table): Table
    {
        $branchId = (int) user_info('branch_id');

        return $table
            ->paginationPageOptions([10, 25, 50, 100])
            ->defaultPaginationPageOption(10)
            ->defaultSort('latest_transaction_at', 'desc')
            ->striped()
            ->filtersLayout(FiltersLayout::Dropdown)

            ->modifyQueryUsing(function (Builder $query) use ($branchId): Builder {
                return $query
                    ->addSelect([
                        // ✅ نفس أسماء _sum حتى تبقى الواجهة سهلة
                        'receivable_total_sum' => function ($q) use ($branchId) {
                            $q->from('client_accounts')
                                ->selectRaw('COALESCE(receivable_total, 0)')
                                ->whereColumn('client_accounts.client_id', 'clients.id')
                                ->where('client_accounts.branch_id', $branchId)
                                ->limit(1);
                        },
                        'receivable_paid_sum' => function ($q) use ($branchId) {
                            $q->from('client_accounts')
                                ->selectRaw('COALESCE(receivable_paid, 0)')
                                ->whereColumn('client_accounts.client_id', 'clients.id')
                                ->where('client_accounts.branch_id', $branchId)
                                ->limit(1);
                        },
                        'receivable_remaining_sum' => function ($q) use ($branchId) {
                            $q->from('client_accounts')
                                ->selectRaw('COALESCE(receivable_remaining, 0)')
                                ->whereColumn('client_accounts.client_id', 'clients.id')
                                ->where('client_accounts.branch_id', $branchId)
                                ->limit(1);
                        },

                        'payable_total_sum' => function ($q) use ($branchId) {
                            $q->from('client_accounts')
                                ->selectRaw('COALESCE(payable_total, 0)')
                                ->whereColumn('client_accounts.client_id', 'clients.id')
                                ->where('client_accounts.branch_id', $branchId)
                                ->limit(1);
                        },
                        'payable_paid_sum' => function ($q) use ($branchId) {
                            $q->from('client_accounts')
                                ->selectRaw('COALESCE(payable_paid, 0)')
                                ->whereColumn('client_accounts.client_id', 'clients.id')
                                ->where('client_accounts.branch_id', $branchId)
                                ->limit(1);
                        },
                        'payable_remaining_sum' => function ($q) use ($branchId) {
                            $q->from('client_accounts')
                                ->selectRaw('COALESCE(payable_remaining, 0)')
                                ->whereColumn('client_accounts.client_id', 'clients.id')
                                ->where('client_accounts.branch_id', $branchId)
                                ->limit(1);
                        },

                        'latest_transaction_at' => function ($q) use ($branchId) {
                            $q->from('client_accounts')
                                ->selectRaw('last_transaction_at')
                                ->whereColumn('client_accounts.client_id', 'clients.id')
                                ->where('client_accounts.branch_id', $branchId)
                                ->limit(1);
                        },

                        // ✅ عدد الأقساط غير المدفوعة (حسب الفرع)
                        'pending_installments_count' => function ($q) use ($branchId) {
                            $q->from('installment_schedules as s')
                                ->join('client_installment_plans as p', 'p.id', '=', 's.installment_plan_id')
                                ->selectRaw('COUNT(*)')
                                ->whereColumn('p.client_id', 'clients.id')
                                ->where('p.branch_id', $branchId)
                                ->whereRaw("COALESCE(s.status,'') <> 'paid'");
                        },
                    ])
                    // ✅ المكتملة: لا مستحق لنا ولا مستحق لهم + بدون أقساط معلّقة
                    ->whereRaw(
                        '(SELECT COALESCE(receivable_remaining,0)
                           FROM client_accounts
                          WHERE client_accounts.client_id = clients.id
                            AND client_accounts.branch_id = ?
                          LIMIT 1) = 0',
                        [$branchId]
                    )
                    ->whereRaw(
                        '(SELECT COALESCE(payable_remaining,0)
                           FROM client_accounts
                          WHERE client_accounts.client_id = clients.id
                            AND client_accounts.branch_id = ?
                          LIMIT 1) = 0',
                        [$branchId]
                    )
                    ->whereRaw(
                        '(SELECT COUNT(*)
                           FROM installment_schedules s
                           JOIN client_installment_plans p ON p.id = s.installment_plan_id
                          WHERE p.client_id = clients.id
                            AND p.branch_id = ?
                            AND COALESCE(s.status,"") <> "paid") = 0',
                        [$branchId]
                    );
            })

            ->columns([
                TextColumn::make('name')
                    ->label('العميل')
                    ->searchable()
                    ->sortable()
                    ->weight('bold')
                    ->icon('heroicon-o-user')
                    ->iconColor('primary'),

                // ✅ صافي = 0 دائماً لكن نعرضه بشكل واضح
                TextColumn::make('net_balance')
                    ->label('الصافي')
                    ->alignEnd()
                    ->state(fn ($record) => 0)
                    ->badge()
                    ->size('lg')
                    ->weight('bold')
                    ->color('success')
                    ->icon('heroicon-o-check-badge')
                    ->formatStateUsing(fn () => '0 د.ع'),

                TextColumn::make('receivable_total_sum')
                    ->label('إجمالي المبيعات')
                    ->alignEnd()
                    ->sortable()
                    ->badge()
                    ->color('gray')
                    ->icon('heroicon-o-shopping-cart')
                    ->iconColor('success')
                    ->formatStateUsing(fn ($state) => number_format((int) ($state ?? 0), 0, '.', ',').' د.ع')
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('receivable_paid_sum')
                    ->label('المدفوع من المبيعات')
                    ->alignEnd()
                    ->sortable()
                    ->badge()
                    ->color('success')
                    ->icon('heroicon-o-banknotes')
                    ->iconColor('success')
                    ->formatStateUsing(fn ($state) => number_format((int) ($state ?? 0), 0, '.', ',').' د.ع')
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('payable_total_sum')
                    ->label('إجمالي المشتريات')
                    ->alignEnd()
                    ->sortable()
                    ->badge()
                    ->color('gray')
                    ->icon('heroicon-o-shopping-bag')
                    ->iconColor('danger')
                    ->formatStateUsing(fn ($state) => number_format((int) ($state ?? 0), 0, '.', ',').' د.ع')
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('payable_paid_sum')
                    ->label('المدفوع من المشتريات')
                    ->alignEnd()
                    ->sortable()
                    ->badge()
                    ->color('success')
                    ->icon('heroicon-o-banknotes')
                    ->iconColor('danger')
                    ->formatStateUsing(fn ($state) => number_format((int) ($state ?? 0), 0, '.', ',').' د.ع')
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('latest_transaction_at')
                    ->label('آخر حركة')
                    ->dateTime('Y-m-d H:i')
                    ->sortable()
                    ->icon('heroicon-o-clock')
                    ->iconColor('gray')
                    ->since()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])

            ->recordUrl(fn ($record) => ClientResource::getUrl('edit', ['record' => $record]))
            ->recordAction(null)
            ->recordActions([])

            ->filters([
                // فلتر: بدون أقساط (مفيد لو حبيت تفصل “مكتمل محاسبياً” عن “مكتمل مع أقساط”)
                Filter::make('no_installments')
                    ->label('بدون أقساط')
                    ->toggle()
                    ->query(fn (Builder $q) => $q->whereRaw(
                        '(SELECT COUNT(*)
                           FROM installment_schedules s
                           JOIN client_installment_plans p ON p.id = s.installment_plan_id
                          WHERE p.client_id = clients.id
                            AND p.branch_id = ?
                            AND COALESCE(s.status,"") <> "paid") = 0',
                        [$branchId]
                    )),

                // فلتر العميل (بحث سريع)
                SelectFilter::make('client_id')
                    ->label('العميل')
                    ->searchable()
                    ->getSearchResultsUsing(function (string $search) {
                        return Client::query()
                            ->where('name', 'like', "%{$search}%")
                            ->orderBy('name')
                            ->limit(50)
                            ->pluck('name', 'id')
                            ->all();
                    })
                    ->getOptionLabelUsing(fn ($value): ?string => Client::query()->whereKey($value)->value('name')),
            ])

            ->emptyStateHeading('لا توجد سجلات')
            ->emptyStateDescription('لا يوجد أي عميل مكتمل (بدون مستحقات وبلا أقساط معلّقة) ضمن هذا الفرع.')
            ->emptyStateIcon('heroicon-o-check-circle');
    }
}
