<?php

namespace App\Filament\Resources\Finance\Expenses\Tables;

use Carbon\Carbon;
use Filament\Tables;
use Filament\Tables\Table;
use App\Models\ExpenseCategory;
use Filament\Actions\EditAction;
use Filament\Actions\ActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Tables\Filters\Filter;
use Illuminate\Support\Facades\Auth;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;
use Filament\Tables\Columns\TextColumn;
use Filament\Forms\Components\DatePicker;
use Filament\Tables\Filters\SelectFilter;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Filters\TernaryFilter;
use App\Filament\Resources\Orders\OrderResource;

class ExpensesTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->defaultSort('id', 'desc')
            ->columns([
                TextColumn::make('id')
                    ->label('التسلسل')
                    ->sortable()
                    ->toggleable(),

                TextColumn::make('category.name')
                    ->label('التصنيف')
                    ->badge()
                    ->sortable()
                    ->searchable()
                    ->toggleable(),

                TextColumn::make('amount')
                    ->label('المبلغ')
                    ->badge()
                    ->color('danger')
                    ->numeric(decimalPlaces: 0, thousandsSeparator: ',') // تنسيق تلقائي
                    ->suffix(' د.ع')
                    ->sortable()
                    ->toggleable(),

                TextColumn::make('description')
                    ->label('الوصف')
                    ->limit(40)
                    ->toggleable(),

                TextColumn::make('user.name')
                    ->label('المستخدم')
                    ->badge()
                    ->color('info')
                    ->placeholder('—')
                    ->toggleable(),

                TextColumn::make('spent_at')
                    ->label('تاريخ الصرف')
                    ->dateTime('Y-m-d H:i')
                    ->sortable()
                    ->formatStateUsing(function ($state) {
                        if (! $state) return '—';
                        $dt = Carbon::parse($state)->tz(config('app.timezone', 'Asia/Baghdad'));
                        $ampm = $dt->format('A') === 'AM' ? 'ص' : 'م';
                        return $dt->format('Y-m-d h:i ') . $ampm;
                    })
                    ->toggleable(),
            ])
            ->filters([
                // تصفية حسب التصنيف (مفلتر على اشتراك المستخدم)
                SelectFilter::make('expense_category_id')
                    ->label('التصنيف')
                    ->options(function () {
                        $sid = (int) (Auth::user()->branch_id ?? 0);
                        return ExpenseCategory::query()
                            ->where('branch_id', $sid)
                            ->orderBy('name')
                            ->pluck('name', 'id');
                    }),

                // مصروفات الحجوزات فقط / العامة فقط / الكل
                TernaryFilter::make('for_order')
                    ->label('مصروفات الحجوزات؟')
                    ->trueLabel('نعم')
                    ->falseLabel('لا')
                    ->queries(
                        true: fn($q) => $q->whereNotNull('order_id'),
                        false: fn($q) => $q->whereNull('order_id'),
                        blank: fn($q) => $q
                    ),

                // نطاق تاريخ spent_at
                Filter::make('spent_range')
                    ->label('نطاق التاريخ')
                    ->schema([
                        DatePicker::make('from')->label('من'),
                        DatePicker::make('to')->label('إلى'),
                    ])
                    ->query(function ($query, array $data) {
                        return $query
                            ->when($data['from'] ?? null, fn($q, $date) => $q->whereDate('spent_at', '>=', $date))
                            ->when($data['to']   ?? null, fn($q, $date) => $q->whereDate('spent_at', '<=', $date));
                    }),
            ])
            ->searchable() // بحث عام على الأعمدة القابلة للبحث
            ->recordActions([
                ActionGroup::make([
                    EditAction::make(),
                    DeleteAction::make(),
                ])
                    ->label('إجراءات')
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ]);
    }
}
