<?php

namespace App\Filament\Resources\Finance\Expenses\Schemas;

use App\Models\ExpenseCategory;
use Coolsam\Flatpickr\Forms\Components\Flatpickr;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;

class ExpenseForm
{
    public static function configure(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make('تفاصيل المصروف')
                    ->description('أدخل معلومات المصروف بشكل دقيق')
                    ->icon('heroicon-o-banknotes')
                    ->schema([
                        // الصف الأول: التصنيف + المبلغ
                        Select::make('expense_category_id')
                            ->label('التصنيف')
                            ->placeholder('اختر التصنيف')
                            ->searchable()
                            ->preload()
                            ->required()
                            ->native(false)
                            ->prefixIcon('heroicon-m-tag')
                            ->options(
                                fn () => ExpenseCategory::query()
                                    ->where('branch_id', (int) user_info('branch_id'))
                                    ->where('is_active', true)
                                    ->orderBy('name')
                                    ->pluck('name', 'id')
                                    ->toArray()
                            )

                            // ✅ إضافة تصنيف مباشرة من نفس الـ Select
                            ->createOptionForm([
                                TextInput::make('name')
                                    ->label('اسم التصنيف')
                                    ->required()
                                    ->maxLength(190),
                            ])
                            ->createOptionAction(
                                fn (\Filament\Actions\Action $action) => $action
                                    ->modalHeading('إضافة تصنيف مصروف جديد')
                                    ->modalSubmitActionLabel('حفظ')
                                    ->modalWidth('md')
                                    ->color('success')
                                    ->icon('heroicon-o-plus')
                            )
                            ->createOptionUsing(function (array $data) {
                                $branchId = (int) user_info('branch_id');
                                $name = trim((string) ($data['name'] ?? ''));

                                if ($name === '') {
                                    return null;
                                }

                                $category = ExpenseCategory::firstOrCreate(
                                    [
                                        'branch_id' => $branchId,
                                        'name' => $name,
                                    ],
                                    [
                                        'is_active' => (bool) ($data['is_active'] ?? true),
                                    ]
                                );

                                return (int) $category->id;
                            })

                            ->columnSpan([
                                'default' => 12,
                                'sm' => 6,
                                'md' => 6,
                                'lg' => 6,
                            ]),

                        TextInput::make('amount')
                            ->label('المبلغ')
                            ->placeholder('0')
                            ->required()
                            ->type('text')
                            ->formatStateUsing(fn ($state) => $state !== null ? number_format((int) $state) : '')
                            ->dehydrateStateUsing(fn ($state) => (int) preg_replace('/[^\d]/', '', (string) $state))
                            ->extraAlpineAttributes([
                                'x-data' => <<<'ALPINE'
                                {
                                    format() {
                                        const raw = $el.value.replace(/[^\d]/g, '');
                                        $el.value = raw ? new Intl.NumberFormat('en-US').format(Number(raw)) : '';
                                    }
                                }
                            ALPINE,
                                'x-on:input' => 'format()',
                                'x-on:blur' => 'format()',
                            ])
                            ->extraAttributes([
                                'inputmode' => 'numeric',
                            ])
                            ->suffix('د.ع')
                            ->prefixIcon('heroicon-m-currency-dollar')
                            ->columnSpan([
                                'default' => 12,
                                'sm' => 6,
                                'md' => 6,
                                'lg' => 6,
                            ]),

                        // الصف الثاني: تاريخ الصرف
                        Flatpickr::make('spent_at')
                            ->label('تاريخ الصرف')
                            ->placeholder('اختر التاريخ')
                            ->format('Y-m-d')
                            ->displayFormat('Y-m-d')
                            ->time(false)
                            ->default(now())
                            ->prefixIcon('heroicon-m-calendar-days')
                            ->columnSpan([
                                'default' => 12,
                                'sm' => 12,
                                'md' => 12,
                                'lg' => 12,
                            ]),

                        // الصف الثالث: الوصف
                        Textarea::make('description')
                            ->label('الوصف')
                            ->placeholder('أضف وصفاً تفصيلياً للمصروف...')
                            ->rows(4)
                            ->autosize()
                            ->minLength(10)
                            ->maxLength(500)
                            ->columnSpanFull(),
                    ])
                    ->columns([
                        'default' => 12,
                        'sm' => 12,
                        'md' => 12,
                        'lg' => 12,
                    ])
                    ->compact()
                    ->columnSpanFull()
                    ->collapsible(),
            ]);
    }
}
