<?php

namespace App\Filament\Resources\Finance\ExpenseCategories\Tables;

use Carbon\Carbon;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Actions\EditAction;
use Filament\Actions\ActionGroup;
use Filament\Actions\CreateAction;
use Filament\Actions\DeleteAction;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\TernaryFilter;

class ExpenseCategoriesTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->defaultSort('id', 'desc')
            ->columns([
                TextColumn::make('index')
                    ->label('التسلسل')
                    ->toggleable()
                    ->rowIndex(),

                TextColumn::make('name')
                    ->label('التصنيف')
                    ->searchable()
                    ->sortable(),

                IconColumn::make('is_active')
                    ->label('الحالة')
                    ->boolean() // يعرض ✓ / ✗ تلقائياً
                    ->trueColor('success')
                    ->falseColor('secondary')
                    ->tooltip(fn($state) => $state ? 'مفعّل' : 'موقوف')
                    ->toggleable()
                    ->extraHeaderAttributes(['style' => 'width:120px']),

                TextColumn::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->formatStateUsing(function ($state) {
                        if (! $state) return '—';
                        $dt = Carbon::parse($state)->tz(config('app.timezone', 'Asia/Baghdad'));
                        $ampm = $dt->format('A') === 'AM' ? 'ص' : 'م';
                        return $dt->format('Y-m-d h:i ') . $ampm;
                    })
                    ->sortable()
                    ->toggleable(),
            ])

            ->filters([
                TernaryFilter::make('is_active')
                    ->label('الحالة')
                    ->trueLabel('مفعّل')
                    ->falseLabel('موقوف')
                    ->native(false), // يخليها كبادج بدلاً من select بسيط
            ])

            // أزرار لكل سجل
            ->recordActions([
                ActionGroup::make([
                    EditAction::make(),
                    DeleteAction::make(),
                ]),
            ])

            // حالة الجدول الفارغ
            ->emptyStateHeading('لا توجد تصنيفات بعد')
            ->emptyStateDescription('ابدأ بإضافة أول تصنيف للمصروفات.')
            ->emptyStateActions([
                CreateAction::make()
                    ->label('إضافة تصنيف')
                    ->mutateDataUsing(function (array $data): array {
                        $data['branch_id'] = (int) (user_info('branch_id') ?? 0);
                        return $data;
                    }),
            ])

            // إجراءات جماعية
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ]);
    }
}
