<?php

namespace App\Filament\Resources\Finance\Debtors\Widgets;

use App\Models\ClientAccount;
use Filament\Widgets\StatsOverviewWidget as BaseWidget;
use Filament\Widgets\StatsOverviewWidget\Stat;

class DebtorsStatsWidget extends BaseWidget
{
    protected function getStats(): array
    {
        $branchId = function_exists('user_info') ? (int) user_info('branch_id') : null;

        // ✅ جلب إحصائيات الديون من client_accounts
        $debtStats = ClientAccount::query()
            ->when($branchId, fn($q) => $q->where('branch_id', $branchId))
            ->selectRaw('
                COALESCE(SUM(CASE WHEN receivable_remaining > 0 THEN receivable_remaining ELSE 0 END), 0) as total_receivable,
                COALESCE(SUM(CASE WHEN payable_remaining > 0 THEN payable_remaining ELSE 0 END), 0) as total_payable,
                COUNT(DISTINCT CASE WHEN receivable_remaining > 0 THEN client_id END) as clients_receivable_count,
                COUNT(DISTINCT CASE WHEN payable_remaining > 0 THEN client_id END) as clients_payable_count,
                COUNT(DISTINCT CASE WHEN receivable_remaining > 0 OR payable_remaining > 0 THEN client_id END) as total_clients
            ')
            ->first();

        // حساب المبالغ
        $totalReceivable = (int) ($debtStats->total_receivable ?? 0);
        $totalPayable = (int) ($debtStats->total_payable ?? 0);
        $netBalance = $totalReceivable - $totalPayable;
        $clientsReceivable = (int) ($debtStats->clients_receivable_count ?? 0);
        $clientsPayable = (int) ($debtStats->clients_payable_count ?? 0);

        return [
            // 1️⃣ مستحق لنا (من العملاء)
            Stat::make('مستحق لنا', $this->formatAmount($totalReceivable))
                ->description($clientsReceivable > 0 ? "{$clientsReceivable} عميل مدين" : 'لا يوجد عملاء مدينين')
                ->descriptionIcon('heroicon-o-arrow-trending-up')
                ->color($totalReceivable > 0 ? 'success' : 'gray'),

            // 2️⃣ مستحق لهم (للموردين)
            Stat::make('مستحق لهم', $this->formatAmount($totalPayable))
                ->description($clientsPayable > 0 ? "{$clientsPayable} مورد/عميل دائن" : 'لا يوجد موردين دائنين')
                ->descriptionIcon('heroicon-o-arrow-trending-down')
                ->color($totalPayable > 0 ? 'danger' : 'gray'),

            // 3️⃣ الصافي (الفرق)
            Stat::make('الصافي', $this->formatAmount(abs($netBalance)))
                ->description($this->getNetDescription($netBalance))
                ->descriptionIcon($this->getNetIcon($netBalance))
                ->color($this->getNetColor($netBalance)),
        ];
    }

    /**
     * وصف الصافي
     */
    protected function getNetDescription(int $netBalance): string
    {
        if ($netBalance > 0) {
            return '✅ لصالحنا (نستلم)';
        } elseif ($netBalance < 0) {
            return '⚠️ علينا (ندفع)';
        }

        return '⚖️ متوازن تماماً';
    }

    /**
     * أيقونة الصافي
     */
    protected function getNetIcon(int $netBalance): string
    {
        if ($netBalance > 0) {
            return 'heroicon-o-arrow-trending-up';
        } elseif ($netBalance < 0) {
            return 'heroicon-o-arrow-trending-down';
        }

        return 'heroicon-o-scale';
    }

    /**
     * لون الصافي
     */
    protected function getNetColor(int $netBalance): string
    {
        if ($netBalance > 0) {
            return 'success';  // أخضر = لصالحنا
        } elseif ($netBalance < 0) {
            return 'warning';  // برتقالي = علينا ديون
        }

        return 'gray';
    }

    /**
     * تنسيق المبلغ
     */
    protected function formatAmount(int|float $amount): string
    {
        return number_format((int) $amount, 0, '.', ',') . ' د.ع';
    }

    /**
     * عدد الأعمدة
     */
    protected function getColumns(): int
    {
        return 3;
    }
}
