<?php

namespace App\Filament\Resources\Finance\Debtors\Tables;

use App\Filament\Resources\Clients\ClientResource;
use App\Models\Client;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Enums\FiltersLayout;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class DebtorsTable
{
    public static function configure(Table $table): Table
    {
        $branchId = (int) user_info('branch_id');

        return $table
            ->paginationPageOptions([10, 25, 50, 100])
            ->defaultPaginationPageOption(10)
            ->defaultSort('receivable_remaining_sum', 'desc')
            ->striped()

            // ✅ الفلاتر في Dropdown
            ->filtersLayout(FiltersLayout::Dropdown)

            ->modifyQueryUsing(function (Builder $query) use ($branchId): Builder {
                // ✅ نجلب أرقام الحساب من client_accounts (حسب الفرع)
                // ونحافظ على نفس أسماء الأعمدة التي تستخدمها بالواجهة (*_sum) حتى يبقى نفس التصميم شغال.
                return $query
                    ->addSelect([
                        'receivable_total_sum' => function ($q) use ($branchId) {
                            $q->from('client_accounts')
                                ->selectRaw('COALESCE(receivable_total, 0)')
                                ->whereColumn('client_accounts.client_id', 'clients.id')
                                ->where('client_accounts.branch_id', $branchId)
                                ->limit(1);
                        },
                        'receivable_paid_sum' => function ($q) use ($branchId) {
                            $q->from('client_accounts')
                                ->selectRaw('COALESCE(receivable_paid, 0)')
                                ->whereColumn('client_accounts.client_id', 'clients.id')
                                ->where('client_accounts.branch_id', $branchId)
                                ->limit(1);
                        },
                        'receivable_remaining_sum' => function ($q) use ($branchId) {
                            $q->from('client_accounts')
                                ->selectRaw('COALESCE(receivable_remaining, 0)')
                                ->whereColumn('client_accounts.client_id', 'clients.id')
                                ->where('client_accounts.branch_id', $branchId)
                                ->limit(1);
                        },

                        'payable_total_sum' => function ($q) use ($branchId) {
                            $q->from('client_accounts')
                                ->selectRaw('COALESCE(payable_total, 0)')
                                ->whereColumn('client_accounts.client_id', 'clients.id')
                                ->where('client_accounts.branch_id', $branchId)
                                ->limit(1);
                        },
                        'payable_paid_sum' => function ($q) use ($branchId) {
                            $q->from('client_accounts')
                                ->selectRaw('COALESCE(payable_paid, 0)')
                                ->whereColumn('client_accounts.client_id', 'clients.id')
                                ->where('client_accounts.branch_id', $branchId)
                                ->limit(1);
                        },
                        'payable_remaining_sum' => function ($q) use ($branchId) {
                            $q->from('client_accounts')
                                ->selectRaw('COALESCE(payable_remaining, 0)')
                                ->whereColumn('client_accounts.client_id', 'clients.id')
                                ->where('client_accounts.branch_id', $branchId)
                                ->limit(1);
                        },

                        'latest_transaction_at' => function ($q) use ($branchId) {
                            $q->from('client_accounts')
                                ->selectRaw('last_transaction_at')
                                ->whereColumn('client_accounts.client_id', 'clients.id')
                                ->where('client_accounts.branch_id', $branchId)
                                ->limit(1);
                        },

                        // ✅ عدد الأقساط غير المدفوعة (حسب الفرع)
                        'pending_installments_count' => function ($q) use ($branchId) {
                            $q->from('installment_schedules as s')
                                ->join('client_installment_plans as p', 'p.id', '=', 's.installment_plan_id')
                                ->selectRaw('COUNT(*)')
                                ->whereColumn('p.client_id', 'clients.id')
                                ->where('p.branch_id', $branchId)
                                // عدّلها إذا عندك أسماء Status مختلفة
                                ->whereRaw("COALESCE(s.status,'') <> 'paid'");
                        },
                    ])
                    // ✅ افتراضياً: ديون علينا فقط (مستحق لنا)
                    ->whereRaw(
                        '(SELECT COALESCE(receivable_remaining,0)
                           FROM client_accounts
                          WHERE client_accounts.client_id = clients.id
                            AND client_accounts.branch_id = ?
                          LIMIT 1) > 0',
                        [$branchId]
                    );
            })

            ->columns([
                // 1️⃣ العميل
                TextColumn::make('name')
                    ->label('العميل')
                    ->searchable()
                    ->sortable()
                    ->weight('bold')
                    ->icon('heroicon-o-user')
                    ->iconColor('primary'),

                // 2️⃣ مستحق لنا
                TextColumn::make('receivable_remaining_sum')
                    ->label('مستحق لنا')
                    ->alignEnd()
                    ->sortable()
                    ->badge()
                    ->size('md')
                    ->color(fn ($state) => (int) ($state ?? 0) > 0 ? 'success' : 'gray')
                    ->icon(fn ($state) => (int) ($state ?? 0) > 0 ? 'heroicon-o-arrow-trending-up' : null)
                    ->formatStateUsing(fn ($state) => number_format((int) ($state ?? 0), 0, '.', ',').' د.ع'),

                // 3️⃣ مستحق لهم
                TextColumn::make('payable_remaining_sum')
                    ->label('مستحق لهم')
                    ->alignEnd()
                    ->sortable()
                    ->badge()
                    ->size('md')
                    ->color(fn ($state) => (int) ($state ?? 0) > 0 ? 'danger' : 'gray')
                    ->icon(fn ($state) => (int) ($state ?? 0) > 0 ? 'heroicon-o-arrow-trending-down' : null)
                    ->formatStateUsing(fn ($state) => number_format((int) ($state ?? 0), 0, '.', ',').' د.ع'),

                // 4️⃣ الصافي
                TextColumn::make('net_balance')
                    ->label('الصافي')
                    ->alignEnd()
                    ->state(function ($record) {
                        $receivable = (int) ($record->receivable_remaining_sum ?? 0);
                        $payable = (int) ($record->payable_remaining_sum ?? 0);

                        return $receivable - $payable;
                    })
                    ->badge()
                    ->size('lg')
                    ->weight('bold')
                    ->color(function ($record) {
                        $receivable = (int) ($record->receivable_remaining_sum ?? 0);
                        $payable = (int) ($record->payable_remaining_sum ?? 0);
                        $net = $receivable - $payable;

                        if ($net > 0) {
                            return 'success';
                        }
                        if ($net < 0) {
                            return 'warning';
                        }

                        return 'gray';
                    })
                    ->icon(function ($record) {
                        $receivable = (int) ($record->receivable_remaining_sum ?? 0);
                        $payable = (int) ($record->payable_remaining_sum ?? 0);
                        $net = $receivable - $payable;

                        if ($net > 0) {
                            return 'heroicon-o-arrow-trending-up';
                        }
                        if ($net < 0) {
                            return 'heroicon-o-arrow-trending-down';
                        }

                        return 'heroicon-o-scale';
                    })
                    ->formatStateUsing(fn ($state) => number_format(abs((int) $state), 0, '.', ',').' د.ع'),

                // 7️⃣ معلومات إضافية
                TextColumn::make('receivable_total_sum')
                    ->label('إجمالي المبيعات')
                    ->alignEnd()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->formatStateUsing(fn ($state) => number_format((int) ($state ?? 0), 0, '.', ',').' د.ع')
                    ->icon('heroicon-o-shopping-cart')
                    ->iconColor('success'),

                TextColumn::make('receivable_paid_sum')
                    ->label('المدفوع من المبيعات')
                    ->alignEnd()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->formatStateUsing(fn ($state) => number_format((int) ($state ?? 0), 0, '.', ',').' د.ع')
                    ->icon('heroicon-o-banknotes')
                    ->iconColor('success'),

                TextColumn::make('payable_total_sum')
                    ->label('إجمالي المشتريات')
                    ->alignEnd()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->formatStateUsing(fn ($state) => number_format((int) ($state ?? 0), 0, '.', ',').' د.ع')
                    ->icon('heroicon-o-shopping-bag')
                    ->iconColor('danger'),

                TextColumn::make('payable_paid_sum')
                    ->label('المدفوع من المشتريات')
                    ->alignEnd()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->formatStateUsing(fn ($state) => number_format((int) ($state ?? 0), 0, '.', ',').' د.ع')
                    ->icon('heroicon-o-banknotes')
                    ->iconColor('danger'),

                TextColumn::make('latest_transaction_at')
                    ->label('آخر حركة')
                    ->dateTime('Y-m-d H:i')
                    ->sortable()
                    ->icon('heroicon-o-clock')
                    ->iconColor('gray')
                    ->since()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])

            ->recordUrl(fn ($record) => ClientResource::getUrl('edit', ['record' => $record]))
            ->recordAction(null)
            ->recordActions([])

            ->filters([
                // 🔍 فلاتر الديون الأساسية
                Filter::make('has_receivable')
                    ->label('مستحق لنا')
                    ->toggle()
                    ->query(fn (Builder $q) => $q->whereRaw(
                        '(SELECT COALESCE(receivable_remaining,0)
                           FROM client_accounts
                          WHERE client_accounts.client_id = clients.id
                            AND client_accounts.branch_id = ?
                          LIMIT 1) > 0',
                        [$branchId]
                    )),

                Filter::make('has_payable')
                    ->label('مستحق لهم')
                    ->toggle()
                    ->query(fn (Builder $q) => $q->whereRaw(
                        '(SELECT COALESCE(payable_remaining,0)
                           FROM client_accounts
                          WHERE client_accounts.client_id = clients.id
                            AND client_accounts.branch_id = ?
                          LIMIT 1) > 0',
                        [$branchId]
                    )),

                Filter::make('has_installments')
                    ->label('لديه أقساط')
                    ->toggle()
                    ->query(fn (Builder $q) => $q->whereRaw(
                        '(SELECT COUNT(*)
                           FROM installment_schedules s
                           JOIN client_installment_plans p ON p.id = s.installment_plan_id
                          WHERE p.client_id = clients.id
                            AND p.branch_id = ?
                            AND COALESCE(s.status,"") <> "paid") > 0',
                        [$branchId]
                    )),

                // 🔍 فلتر العميل (أداء سريع + بحث)
                SelectFilter::make('client_id')
                    ->label('العميل')
                    ->searchable()
                    ->getSearchResultsUsing(function (string $search): array {
                        return Client::query()
                            ->where('name', 'like', "%{$search}%")
                            ->orderBy('name')
                            ->limit(50)
                            ->pluck('name', 'id')
                            ->all();
                    })
                    ->getOptionLabelUsing(fn ($value): ?string => Client::query()->whereKey($value)->value('name')),

                // 🔍 فلتر حالة الصافي
                SelectFilter::make('net_balance_status')
                    ->label('حالة الصافي')
                    ->options([
                        'in_our_favor' => '✅ لصالحنا',
                        'in_their_favor' => '⚠️ لصالحهم',
                        'balanced' => '⚖️ متوازن',
                    ])
                    ->query(function (Builder $query, array $data) use ($branchId) {
                        $v = $data['value'] ?? null;

                        return match ($v) {
                            'in_our_favor' => $query->whereRaw(
                                '(SELECT COALESCE(receivable_remaining,0) FROM client_accounts WHERE client_accounts.client_id = clients.id AND client_accounts.branch_id = ? LIMIT 1)
                                 >
                                 (SELECT COALESCE(payable_remaining,0) FROM client_accounts WHERE client_accounts.client_id = clients.id AND client_accounts.branch_id = ? LIMIT 1)',
                                [$branchId, $branchId]
                            ),

                            'in_their_favor' => $query->whereRaw(
                                '(SELECT COALESCE(payable_remaining,0) FROM client_accounts WHERE client_accounts.client_id = clients.id AND client_accounts.branch_id = ? LIMIT 1)
                                 >
                                 (SELECT COALESCE(receivable_remaining,0) FROM client_accounts WHERE client_accounts.client_id = clients.id AND client_accounts.branch_id = ? LIMIT 1)',
                                [$branchId, $branchId]
                            ),

                            'balanced' => $query->whereRaw(
                                '(SELECT COALESCE(receivable_remaining,0) FROM client_accounts WHERE client_accounts.client_id = clients.id AND client_accounts.branch_id = ? LIMIT 1)
                                 =
                                 (SELECT COALESCE(payable_remaining,0) FROM client_accounts WHERE client_accounts.client_id = clients.id AND client_accounts.branch_id = ? LIMIT 1)',
                                [$branchId, $branchId]
                            ),

                            default => $query,
                        };
                    }),
            ])

            ->emptyStateHeading('لا توجد ديون')
            ->emptyStateDescription('لا توجد أي مبالغ متبقية على العملاء ضمن هذا الفرع.')
            ->emptyStateIcon('heroicon-o-check-circle');
    }
}
