<?php

namespace App\Filament\Resources\ExchangeRates;

use BackedEnum;
use UnitEnum;
use App\Support\NavGroups;
use Filament\Tables\Table;
use App\Models\ExchangeRate;
use Filament\Schemas\Schema;
use Filament\Actions\EditAction;
use Filament\Resources\Resource;
use Filament\Actions\ActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Support\Icons\Heroicon;
use Filament\Actions\BulkActionGroup;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Toggle;
use Filament\Actions\DeleteBulkAction;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Forms\Components\TextInput;
use Filament\Tables\Filters\SelectFilter;
use App\Filament\Resources\ExchangeRates\Pages\ManageExchangeRates;

class ExchangeRateResource extends Resource
{
    protected static ?string $model = ExchangeRate::class;

    protected static string|BackedEnum|null $navigationIcon = Heroicon::CurrencyDollar;
    protected static string|UnitEnum|null  $navigationGroup = NavGroups::Dashboard;
    protected static ?string $navigationLabel               = 'سعر الصرف';
    protected static ?string $modelLabel                    = 'سعر صرف';
    protected static ?string $pluralModelLabel              = 'سعر الصرف';
    protected static ?int $navigationSort                   = 11;
    protected static ?string $recordTitleAttribute          = 'name';

    public static function canGloballySearch(): bool
    {
        return false;
    }
    public static function form(Schema $schema): Schema
    {
        return $schema
            ->components([
                Select::make('currency_id')
                    ->relationship('currency', 'name')
                    ->label('العملة')
                    ->searchable()
                    ->preload()
                    ->required(),
                TextInput::make('rate')
                    ->label('سعر الصرف')
                    ->required()
                    ->formatStateUsing(fn($state) => $state !== null ? number_format((int) $state) : '')
                    ->dehydrateStateUsing(fn($state) => (int) preg_replace('/[^\d]/', '', (string) $state))
                    ->extraInputAttributes([
                        'inputmode'     => 'numeric',
                        'dir'           => 'rtl',
                        'x-data'        => '{}',
                        'x-on:input'    => <<<'JS'
                            let raw = $el.value.replace(/[^\d]/g, '');
                            $el.value = raw ? new Intl.NumberFormat('en-US').format(raw) : '';
                        JS,
                    ])
                    ->helperText('اكتب كم يساوي 1 من هذه العملة بالعملة الافتراضية (مثلاً 1,420)'),

                Toggle::make('is_active')
                    ->label('الحالة')
                    ->default(true),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('name')
            ->columns([
                TextColumn::make('currency.name')
                    ->label('العملة')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('currency.code')
                    ->label('الرمز')
                    ->sortable(),
                TextColumn::make('rate')
                    ->label('السعر')
                    ->sortable()
                    ->formatStateUsing(function ($state, $record) {
                        if (is_null($state)) return '-';
                        // العملة المرتبطة (مثل USD, IQD)
                        $currencyCode = 'IQD';
                        // رمز العملة إن وجد
                        $symbol = match ($currencyCode) {
                            'IQD' => 'د.ع',
                            default => $currencyCode,
                        };

                        // تنسيق المبلغ مع فواصل آلاف
                        return number_format($state, 0) . ' ' . $symbol;
                    }),

                IconColumn::make('is_active')
                    ->boolean()
                    ->label('الحالة'),
                TextColumn::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('updated_at')
                    ->label('تاريخ التحديث')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //
                SelectFilter::make('currency_id')
                    ->label('العملة')
                    ->relationship('currency', 'name')
                    ->preload()
                    ->searchable(),
                SelectFilter::make('is_active')
                    ->label('الحالة')
                    ->options([
                        true  => 'مفعل',
                        false => 'متوقف',
                    ]),

            ])
            ->recordActions([
                ActionGroup::make([
                    EditAction::make(),
                    DeleteAction::make(),
                ]),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ])
            ->defaultSort('created_at', 'desc');
    }

    public static function getPages(): array
    {
        return [
            'index' => ManageExchangeRates::route('/'),
        ];
    }
}
