<?php

namespace App\Filament\Resources\Coupons\Tables;

use App\Models\Coupon;
use Coolsam\Flatpickr\Forms\Components\Flatpickr;
use Filament\Actions\ActionGroup;
use Filament\Actions\BulkAction;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TernaryFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Carbon;

class CouponsTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('code')
                    ->label('كود الكوبون')
                    ->searchable()
                    ->sortable()
                    ->weight('bold')
                    ->copyable()
                    ->copyMessage('تم نسخ الكود')
                    ->badge()
                    ->color('primary'),

                TextColumn::make('name')
                    ->label('الاسم')
                    ->searchable()
                    ->sortable()
                    ->placeholder('-')
                    ->toggleable(),

                TextColumn::make('type')
                    ->label('النوع')
                    ->badge()
                    ->formatStateUsing(fn (string $state) => match ($state) {
                        'percentage' => 'نسبة مئوية',
                        'fixed' => 'مبلغ ثابت',
                        default => $state,
                    })
                    ->icon(fn (string $state) => match ($state) {
                        'percentage' => 'heroicon-o-receipt-percent',
                        'fixed' => 'heroicon-o-banknotes',
                        default => 'heroicon-o-tag',
                    })
                    ->color(fn (string $state) => match ($state) {
                        'percentage' => 'success',
                        'fixed' => 'info',
                        default => 'gray',
                    })
                    ->sortable(),

                TextColumn::make('discount_value')
                    ->label('قيمة الخصم')
                    ->formatStateUsing(function (Coupon $record): string {
                        if ($record->type === 'percentage') {
                            return $record->discount_value.'%';
                        }

                        return number_format($record->discount_value, 0, '', ',').' د.ع';
                    })
                    ->badge()
                    ->color('success'),

                TextColumn::make('min_purchase_amount')
                    ->label('الحد الأدنى')
                    ->formatStateUsing(fn ($state) => $state ? number_format($state, 0, '', ',').' د.ع' : '-')
                    ->alignEnd()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('used_count')
                    ->label('الاستخدام')
                    ->formatStateUsing(function (Coupon $record): string {
                        if ($record->usage_limit_total) {
                            return $record->used_count.'/'.$record->usage_limit_total;
                        }

                        return $record->used_count.' (∞)';
                    })
                    ->badge()
                    ->color(fn (Coupon $record) => match (true) {
                        $record->usage_limit_total && $record->used_count >= $record->usage_limit_total => 'danger',
                        $record->usage_limit_total && $record->used_count >= ($record->usage_limit_total * 0.8) => 'warning',
                        default => 'gray',
                    })
                    ->alignCenter(),

                IconColumn::make('is_active')
                    ->label('مفعّل')
                    ->boolean()
                    ->alignCenter()
                    ->sortable(),

                TextColumn::make('is_valid')
                    ->label('الحالة')
                    ->badge()
                    ->formatStateUsing(fn (Coupon $record) => $record->is_valid ? 'ساري' : 'غير ساري')
                    ->color(fn (Coupon $record) => $record->is_valid ? 'success' : 'danger')
                    ->icon(fn (Coupon $record) => $record->is_valid ? 'heroicon-o-check-circle' : 'heroicon-o-x-circle'),

                TextColumn::make('starts_at')
                    ->label('يبدأ')
                    ->dateTime('Y-m-d')
                    ->sortable()
                    ->placeholder('فوري')
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('ends_at')
                    ->label('ينتهي')
                    ->dateTime('Y-m-d')
                    ->sortable()
                    ->placeholder('بدون نهاية')
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->dateTime('Y-m-d')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                SelectFilter::make('type')
                    ->label('النوع')
                    ->options([
                        'percentage' => 'نسبة مئوية',
                        'fixed' => 'مبلغ ثابت',
                    ])
                    ->multiple(),

                TernaryFilter::make('is_active')
                    ->label('الحالة')
                    ->placeholder('الكل')
                    ->trueLabel('مفعّل')
                    ->falseLabel('معطّل'),

                Filter::make('valid')
                    ->label('ساري المفعول')
                    ->query(fn (Builder $query) => $query->valid()),

                Filter::make('expired')
                    ->label('منتهي')
                    ->query(fn (Builder $query) => $query
                        ->where('is_active', true)
                        ->where('ends_at', '<', now())),

                Filter::make('exhausted')
                    ->label('مستنفد')
                    ->query(fn (Builder $query) => $query
                        ->whereNotNull('usage_limit_total')
                        ->whereColumn('used_count', '>=', 'usage_limit_total')),

                Filter::make('date_range')
                    ->form([
                        Flatpickr::make('from_date')
                            ->label('من تاريخ')
                            ->displayFormat('Y-m-d')
                            ->native(false),
                        Flatpickr::make('to_date')
                            ->label('إلى تاريخ')
                            ->displayFormat('Y-m-d')
                            ->native(false),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['from_date'] ?? null,
                                fn (Builder $q, $date) => $q->whereDate('created_at', '>=', Carbon::parse($date))
                            )
                            ->when(
                                $data['to_date'] ?? null,
                                fn (Builder $q, $date) => $q->whereDate('created_at', '<=', Carbon::parse($date))
                            );
                    })
                    ->indicateUsing(function (array $data): array {
                        $indicators = [];
                        if ($data['from_date'] ?? null) {
                            $indicators['from_date'] = 'من: '.$data['from_date'];
                        }
                        if ($data['to_date'] ?? null) {
                            $indicators['to_date'] = 'إلى: '.$data['to_date'];
                        }

                        return $indicators;
                    }),
            ])
            ->recordActions([
                ActionGroup::make([
                    EditAction::make(),
                    DeleteAction::make()
                        ->requiresConfirmation()
                        ->modalHeading('حذف الكوبون')
                        ->modalDescription('هل أنت متأكد من حذف هذا الكوبون؟'),
                ]),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make()
                        ->requiresConfirmation()
                        ->modalHeading('حذف الكوبونات المحددة')
                        ->modalDescription('هل أنت متأكد من حذف الكوبونات المحددة؟'),

                    BulkAction::make('activate')
                        ->label('تفعيل')
                        ->icon('heroicon-o-check-circle')
                        ->color('success')
                        ->requiresConfirmation()
                        ->modalHeading('تفعيل الكوبونات')
                        ->modalDescription('سيتم تفعيل جميع الكوبونات المحددة')
                        ->action(fn ($records) => $records->each->update(['is_active' => true]))
                        ->deselectRecordsAfterCompletion(),

                    BulkAction::make('deactivate')
                        ->label('إلغاء التفعيل')
                        ->icon('heroicon-o-x-circle')
                        ->color('danger')
                        ->requiresConfirmation()
                        ->modalHeading('إلغاء تفعيل الكوبونات')
                        ->modalDescription('سيتم إلغاء تفعيل جميع الكوبونات المحددة')
                        ->action(fn ($records) => $records->each->update(['is_active' => false]))
                        ->deselectRecordsAfterCompletion(),
                ]),
            ])
            ->defaultSort('created_at', 'desc')
            ->emptyStateHeading('لا توجد كوبونات')
            ->emptyStateDescription('ابدأ بإضافة كوبون جديد')
            ->emptyStateIcon('heroicon-o-ticket')
            ->striped();
    }
}
