<?php

namespace App\Filament\Resources\Coupons\Schemas;

use App\Models\Coupon;
use Filament\Actions\Action;
use Filament\Forms\Components\DateTimePicker;
use Filament\Forms\Components\Placeholder;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Schema;
use Illuminate\Support\HtmlString;
use Illuminate\Support\Str;

class CouponForm
{
    public static function configure(Schema $schema): Schema
    {
        return $schema->components([
            self::sectionBasicInfo(),
            self::sectionDiscountSettings(),
            self::sectionUsageLimits(),
            self::sectionSchedule(),
        ]);
    }

    /* ================== Helpers ================== */

    protected static function formatMoney(mixed $value): string
    {
        return number_format((int) $value, 0, '', ',');
    }

    protected static function toInt(mixed $value): int
    {
        if ($value === null || $value === '') {
            return 0;
        }

        $value = (string) $value;
        $clean = preg_replace('/[^\d]/u', '', $value);

        return (int) ($clean ?: 0);
    }

    protected static function moneyInput(string $name, string $label, bool $required = false): TextInput
    {
        $input = TextInput::make($name)
            ->label($label)
            ->inputMode('numeric')
            ->step(1)
            ->default('0')
            ->live(onBlur: true)
            ->afterStateHydrated(function ($state, callable $set) use ($name) {
                $set($name, self::formatMoney($state));
            })
            ->afterStateUpdated(function ($state, callable $set) use ($name) {
                $set($name, self::formatMoney($state));
            })
            ->dehydrateStateUsing(fn ($state) => self::toInt($state))
            ->rule('regex:/^[\d,]+$/u');

        return $required ? $input->required() : $input;
    }

    /* ================== المعلومات الأساسية ================== */

    protected static function sectionBasicInfo(): Section
    {
        return Section::make('معلومات الكوبون')
            ->description('البيانات الأساسية للكوبون')
            ->icon('heroicon-o-ticket')
            ->collapsible()
            ->columnSpanFull()
            ->schema([
                Grid::make(2)->schema([
                    TextInput::make('code')
                        ->label('كود الكوبون')
                        ->required()
                        ->maxLength(50)
                        ->placeholder('مثال: SUMMER2024')
                        ->unique(Coupon::class, 'code', ignoreRecord: true)
                        ->extraInputAttributes(['style' => 'text-transform: uppercase'])
                        ->helperText('سيتم تحويله تلقائياً إلى أحرف كبيرة')
                        ->prefixIcon('heroicon-o-hashtag')
                        ->suffixAction(
                            Action::make('generate')
                                ->icon('heroicon-o-sparkles')
                                ->tooltip('توليد كود عشوائي')
                                ->action(function (callable $set) {
                                    $code = strtoupper(Str::random(8));
                                    $set('code', $code);
                                })
                        ),

                    TextInput::make('name')
                        ->label('اسم الكوبون')
                        ->maxLength(255)
                        ->placeholder('مثال: خصم الصيف')
                        ->helperText('اسم داخلي للتعريف بالكوبون')
                        ->prefixIcon('heroicon-o-tag'),

                    Textarea::make('description')
                        ->label('الوصف')
                        ->rows(2)
                        ->placeholder('وصف الكوبون...')
                        ->columnSpanFull(),

                    Toggle::make('is_active')
                        ->label('مفعّل')
                        ->default(true)
                        ->inline(false),
                ]),
            ]);
    }

    /* ================== إعدادات الخصم ================== */

    protected static function sectionDiscountSettings(): Section
    {
        return Section::make('إعدادات الخصم')
            ->description('نوع ومقدار الخصم')
            ->icon('heroicon-o-receipt-percent')
            ->collapsible()
            ->columnSpanFull()
            ->schema([
                Grid::make(3)->schema([
                    Select::make('type')
                        ->label('نوع الخصم')
                        ->options([
                            'percentage' => 'نسبة مئوية',
                            'fixed' => 'مبلغ ثابت',
                        ])
                        ->required()
                        ->default('percentage')
                        ->live()
                        ->prefixIcon('heroicon-o-sparkles'),

                    TextInput::make('discount_value')
                        ->label(fn (Get $get) => $get('type') === 'percentage' ? 'نسبة الخصم %' : 'مبلغ الخصم')
                        ->numeric()
                        ->required()
                        ->minValue(0)
                        ->maxValue(fn (Get $get) => $get('type') === 'percentage' ? 100 : null)
                        ->suffix(fn (Get $get) => $get('type') === 'percentage' ? '%' : 'د.ع')
                        ->prefixIcon('heroicon-o-calculator'),

                    self::moneyInput('min_purchase_amount', 'الحد الأدنى للشراء')
                        ->helperText('اترك فارغاً لعدم وجود حد أدنى'),

                    self::moneyInput('max_discount_amount', 'الحد الأقصى للخصم')
                        ->helperText('اترك فارغاً لعدم وجود حد أقصى')
                        ->visible(fn (Get $get) => $get('type') === 'percentage'),
                ]),

                Placeholder::make('discount_preview')
                    ->hiddenLabel()
                    ->content(function (Get $get): HtmlString {
                        $type = $get('type');
                        $value = $get('discount_value');
                        $minPurchase = self::toInt($get('min_purchase_amount'));
                        $maxDiscount = self::toInt($get('max_discount_amount'));

                        if (! $value) {
                            return new HtmlString('');
                        }

                        $description = match ($type) {
                            'percentage' => "خصم {$value}% على الطلب",
                            'fixed' => 'خصم '.number_format($value, 0, '', ',').' د.ع',
                            default => '',
                        };

                        if ($minPurchase > 0) {
                            $description .= ' | الحد الأدنى للطلب: '.self::formatMoney($minPurchase).' د.ع';
                        }

                        if ($type === 'percentage' && $maxDiscount > 0) {
                            $description .= ' | الحد الأقصى للخصم: '.self::formatMoney($maxDiscount).' د.ع';
                        }

                        return new HtmlString(
                            '<div class="rounded-lg border-r-4 p-3 bg-green-50 border-green-500 text-green-800 dark:bg-green-900/40 dark:border-green-400 dark:text-green-100">
                                <span class="font-semibold">📋 ملخص: </span>'.$description.'
                            </div>'
                        );
                    }),
            ]);
    }

    /* ================== حدود الاستخدام ================== */

    protected static function sectionUsageLimits(): Section
    {
        return Section::make('حدود الاستخدام')
            ->description('التحكم في عدد مرات استخدام الكوبون')
            ->icon('heroicon-o-chart-bar')
            ->collapsible()
            ->collapsed()
            ->columnSpanFull()
            ->schema([
                Grid::make(3)->schema([
                    TextInput::make('usage_limit_total')
                        ->label('الحد الأقصى الإجمالي')
                        ->numeric()
                        ->minValue(1)
                        ->placeholder('غير محدود')
                        ->helperText('إجمالي مرات الاستخدام المسموحة')
                        ->prefixIcon('heroicon-o-user-group'),

                    TextInput::make('usage_limit_per_client')
                        ->label('الحد لكل عميل')
                        ->numeric()
                        ->minValue(1)
                        ->placeholder('غير محدود')
                        ->helperText('عدد مرات الاستخدام لكل عميل')
                        ->prefixIcon('heroicon-o-user'),

                    TextInput::make('used_count')
                        ->label('عدد مرات الاستخدام')
                        ->numeric()
                        ->default(0)
                        ->disabled()
                        ->dehydrated(false)
                        ->prefixIcon('heroicon-o-calculator')
                        ->visibleOn('edit'),
                ]),

                Placeholder::make('usage_stats')
                    ->hiddenLabel()
                    ->content(function (Get $get): HtmlString {
                        $limit = $get('usage_limit_total');
                        $count = $get('used_count') ?? 0;
                        $perClient = $get('usage_limit_per_client');

                        $stats = [];

                        if ($limit) {
                            $remaining = max(0, (int) $limit - (int) $count);
                            $percentage = round(($count / $limit) * 100);
                            $stats[] = "الاستخدامات: {$count}/{$limit} ({$percentage}%)";

                            if ($remaining === 0) {
                                return new HtmlString(
                                    '<div class="rounded-lg border-r-4 p-3 bg-red-50 border-red-500 text-red-800 dark:bg-red-900/40 dark:text-red-100">
                                        <span class="font-semibold">❌ الكوبون استُنفد!</span> تم استخدام جميع المرات المتاحة.
                                    </div>'
                                );
                            }
                        } else {
                            $stats[] = "الاستخدامات: {$count} (غير محدود)";
                        }

                        if ($perClient) {
                            $stats[] = "الحد لكل عميل: {$perClient}";
                        }

                        return new HtmlString(
                            '<div class="rounded-lg border-r-4 p-3 bg-blue-50 border-blue-500 text-blue-800 dark:bg-blue-900/40 dark:text-blue-100">
                                <span class="font-semibold">📊 إحصائيات: </span>'.implode(' | ', $stats).'
                            </div>'
                        );
                    })
                    ->visibleOn('edit'),
            ]);
    }

    /* ================== الجدولة ================== */

    protected static function sectionSchedule(): Section
    {
        return Section::make('جدولة الكوبون')
            ->description('تحديد فترة صلاحية الكوبون')
            ->icon('heroicon-o-calendar')
            ->collapsible()
            ->collapsed()
            ->columnSpanFull()
            ->schema([
                Grid::make(2)->schema([
                    DateTimePicker::make('starts_at')
                        ->label('يبدأ في')
                        ->placeholder('فوري')
                        ->helperText('اترك فارغاً للبدء فوراً')
                        ->prefixIcon('heroicon-o-play'),

                    DateTimePicker::make('ends_at')
                        ->label('ينتهي في')
                        ->placeholder('بدون نهاية')
                        ->helperText('اترك فارغاً لعدم وجود نهاية')
                        ->afterOrEqual('starts_at')
                        ->prefixIcon('heroicon-o-stop'),
                ]),

                Placeholder::make('validity_status')
                    ->hiddenLabel()
                    ->content(function (Get $get): HtmlString {
                        $startsAt = $get('starts_at');
                        $endsAt = $get('ends_at');
                        $isActive = $get('is_active');

                        if (! $isActive) {
                            return new HtmlString(
                                '<div class="rounded-lg border-r-4 p-3 bg-gray-100 border-gray-400 text-gray-600 dark:bg-gray-800 dark:text-gray-300">
                                    <span class="font-semibold">⏸️ الكوبون معطّل</span>
                                </div>'
                            );
                        }

                        if ($startsAt && now()->lt($startsAt)) {
                            return new HtmlString(
                                '<div class="rounded-lg border-r-4 p-3 bg-amber-50 border-amber-500 text-amber-800 dark:bg-amber-900/40 dark:text-amber-100">
                                    <span class="font-semibold">⏳ الكوبون لم يبدأ بعد</span>
                                </div>'
                            );
                        }

                        if ($endsAt && now()->gt($endsAt)) {
                            return new HtmlString(
                                '<div class="rounded-lg border-r-4 p-3 bg-red-50 border-red-500 text-red-800 dark:bg-red-900/40 dark:text-red-100">
                                    <span class="font-semibold">❌ الكوبون منتهي</span>
                                </div>'
                            );
                        }

                        return new HtmlString(
                            '<div class="rounded-lg border-r-4 p-3 bg-green-50 border-green-500 text-green-800 dark:bg-green-900/40 dark:text-green-100">
                                <span class="font-semibold">✅ الكوبون ساري المفعول</span>
                            </div>'
                        );
                    })
                    ->visibleOn('edit'),
            ]);
    }
}
