<?php

namespace App\Filament\Resources\ClientReceipts\Tables;

use Carbon\Carbon;
use Filament\Tables\Table;
use App\Enums\ReceiptStatus;
use App\Models\ClientReceipt;
use Filament\Actions\EditAction;
use Filament\Actions\ViewAction;
use Filament\Actions\ActionGroup;
use Filament\Tables\Filters\Filter;
use Filament\Actions\BulkActionGroup;
use Filament\Forms\Components\Select;
use Filament\Actions\DeleteBulkAction;
use Filament\Support\Enums\FontWeight;
use Filament\Actions\RestoreBulkAction;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Forms\Components\DatePicker;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Columns\BooleanColumn;

use Filament\Tables\Filters\TrashedFilter;
use Filament\Actions\ForceDeleteBulkAction;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Components\Utilities\Set;

class ClientReceiptsTable
{
    public static function configure(Table $table): Table
    {
        $iqd = fn($v) => number_format((int) ($v ?? 0), 0, '.', ',');
        $na  = '—';

        return $table
            ->defaultSort('receipt_date', 'desc')

            ->columns([
                TextColumn::make('receipt_number')
                    ->label('رقم الوصل')
                    ->searchable()
                    ->sortable()
                    ->weight(FontWeight::Bold)
                    ->placeholder($na),

                TextColumn::make('client.name')
                    ->label('العميل')
                    ->searchable()
                    ->placeholder($na),


                // ====== ملخص مالي للوصل ======
                TextColumn::make('total_amount')
                    ->label('إجمالي الوصل')
                    ->suffix(' د.ع')
                    ->color('info')
                    ->weight(FontWeight::Bold)
                    ->formatStateUsing(fn($state) => $iqd($state))
                    ->tooltip('المبلغ الإجمالي لهذا الوصل'),

                TextColumn::make('total_paid')
                    ->label('المبلغ المدفوع')
                    ->suffix(' د.ع')
                    ->color('success')
                    ->formatStateUsing(fn($state) => $iqd($state))
                    ->tooltip('إجمالي ما تم تخصيصه من هذا الوصل لسداد الفواتير'),

                TextColumn::make('total_remaining')
                    ->label('متبقي بعد الدفع')
                    ->suffix(' د.ع')
                    ->color(fn($state) => ((int) $state === 0 ? 'success' : 'warning'))
                    ->formatStateUsing(fn($state) => $iqd($state))
                    ->tooltip('المتبقي من الوصل بعد التخصيص على الفواتير'),

                TextColumn::make('payment_type')
                    ->label('طريقة الدفع')
                    ->badge()
                    ->color(fn(?string $state) => match ($state) {
                        'cash'     => 'success',
                        'transfer' => 'info',
                        'card'     => 'warning',
                        default    => 'gray',
                    })
                    ->formatStateUsing(fn(?string $state) => match ($state) {
                        'cash'     => 'نقدًا',
                        'transfer' => 'تحويل',
                        'card'     => 'بطاقة',
                        null, ''   => $na,
                        default    => $state,
                    })
                    ->placeholder($na),

                IconColumn::make('status')
                    ->label('الحالة')
                    ->icon(function ($state) {
                        $value = $state instanceof ReceiptStatus ? $state->value : (string) $state;

                        return match ($value) {
                            ReceiptStatus::ISSUED->value   => 'heroicon-o-check-circle',
                            ReceiptStatus::POSTED->value   => 'heroicon-o-check-badge',
                            ReceiptStatus::CANCELED->value => 'heroicon-o-x-circle',
                            ReceiptStatus::REVERSED->value => 'heroicon-o-arrow-uturn-left',
                            default                        => 'heroicon-o-question-mark-circle',
                        };
                    })
                    ->color(function ($state) {
                        if ($state instanceof ReceiptStatus) {
                            return $state->color();
                        }

                        $value = (string) $state;

                        return match ($value) {
                            ReceiptStatus::ISSUED->value   => ReceiptStatus::ISSUED->color(),
                            ReceiptStatus::POSTED->value   => ReceiptStatus::POSTED->color(),
                            ReceiptStatus::CANCELED->value => ReceiptStatus::CANCELED->color(),
                            ReceiptStatus::REVERSED->value => ReceiptStatus::REVERSED->color(),
                            default                        => 'gray',
                        };
                    })
                    ->tooltip(function ($state) {
                        if ($state instanceof ReceiptStatus) {
                            // نستخدم الـ label من الـ Enum
                            return $state->label();
                        }

                        $value = (string) $state;

                        return match ($value) {
                            ReceiptStatus::ISSUED->value   => ReceiptStatus::ISSUED->label(),
                            ReceiptStatus::POSTED->value   => ReceiptStatus::POSTED->label(),
                            ReceiptStatus::CANCELED->value => ReceiptStatus::CANCELED->label(),
                            ReceiptStatus::REVERSED->value => ReceiptStatus::REVERSED->label(),
                            default                        => 'غير معروف',
                        };
                    })
                    ->sortable()
                    ->placeholder($na),

                TextColumn::make('user.name')
                    ->label('الموظف')
                    ->placeholder($na),

                TextColumn::make('receipt_date')
                    ->label('تاريخ الوصل')
                    ->date('Y-m-d')
                    ->sortable()
                    ->placeholder($na),

                // مخفية افتراضيًا
                TextColumn::make('created_at')
                    ->label('أُنشئ في')
                    ->dateTime('Y-m-d H:i')
                    ->sortable()
                    ->placeholder($na)
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('updated_at')
                    ->label('عُدّل في')
                    ->dateTime('Y-m-d H:i')
                    ->sortable()
                    ->placeholder($na)
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('deleted_at')
                    ->label('حُذف في')
                    ->dateTime('Y-m-d H:i')
                    ->sortable()
                    ->placeholder($na)
                    ->toggleable(isToggledHiddenByDefault: true),
            ])

            ->filters([
                SelectFilter::make('client_id')
                    ->label('العميل')
                    ->relationship('client', 'name')
                    ->searchable()
                    ->preload(),

                Filter::make('date_period')
                    ->label('الفترة')
                    ->schema([
                        Select::make('preset')
                            ->label('نوع الفترة')
                            ->native(false)
                            ->options([
                                'today'      => 'اليوم',
                                'yesterday'  => 'أمس',
                                'this_week'  => 'هذا الأسبوع',
                                'last_week'  => 'الأسبوع الماضي',
                                'this_month' => 'هذا الشهر',
                                'custom'     => 'مخصص',
                            ])
                            ->placeholder('اختر فترة')
                            ->live()
                            ->afterStateUpdated(function (Set $set, ?string $state) {
                                if ($state !== 'custom') {
                                    $set('from', null);
                                    $set('to', null);
                                }
                            }),

                        DatePicker::make('from')
                            ->label('من')
                            ->native(false)
                            ->displayFormat('Y-m-d')
                            ->placeholder('من تاريخ')
                            ->prefixIcon('heroicon-o-calendar-days')
                            ->visible(fn(Get $get) => $get('preset') === 'custom')
                            ->required(fn(Get $get) => $get('preset') === 'custom'),

                        DatePicker::make('to')
                            ->label('إلى')
                            ->native(false)
                            ->displayFormat('Y-m-d')
                            ->placeholder('إلى تاريخ')
                            ->prefixIcon('heroicon-o-calendar-days')
                            ->visible(fn(Get $get) => $get('preset') === 'custom')
                            ->required(fn(Get $get) => $get('preset') === 'custom'),
                    ])
                    ->indicateUsing(function (array $data): array {
                        $badges = [];
                        $map = [
                            'today'      => 'اليوم',
                            'yesterday'  => 'أمس',
                            'this_week'  => 'هذا الأسبوع',
                            'last_week'  => 'الأسبوع الماضي',
                            'this_month' => 'هذا الشهر',
                            'custom'     => 'مخصص',
                        ];
                        if (($data['preset'] ?? null) && $data['preset'] !== 'custom') {
                            $badges[] = $map[$data['preset']] ?? 'الفترة';
                        } elseif (($data['preset'] ?? null) === 'custom') {
                            if (!empty($data['from'])) $badges[] = 'من: ' . $data['from'];
                            if (!empty($data['to']))   $badges[] = 'إلى: ' . $data['to'];
                        }
                        return $badges;
                    })
                    ->query(function ($query, array $data) {
                        $preset = $data['preset'] ?? null;
                        $now = Carbon::now();

                        $range = match ($preset) {
                            'today'      => [$now->copy()->startOfDay(),          $now->copy()->endOfDay()],
                            'yesterday'  => [$now->copy()->subDay()->startOfDay(), $now->copy()->subDay()->endOfDay()],
                            'this_week'  => [$now->copy()->startOfWeek(),         $now->copy()->endOfWeek()],
                            'last_week'  => [$now->copy()->subWeek()->startOfWeek(), $now->copy()->subWeek()->endOfWeek()],
                            'this_month' => [$now->copy()->startOfMonth(),        $now->copy()->endOfMonth()],
                            default      => null,
                        };

                        if ($preset === 'custom') {
                            return $query
                                ->when($data['from'] ?? null, fn($q, $d) => $q->whereDate('receipt_date', '>=', $d))
                                ->when($data['to']   ?? null, fn($q, $d) => $q->whereDate('receipt_date', '<=', $d));
                        }

                        if ($range) {
                            [$start, $end] = $range;
                            return $query->whereBetween('receipt_date', [$start->toDateString(), $end->toDateString()]);
                        }

                        return $query;
                    }),

                TrashedFilter::make()->label('المحذوفة'),
            ])

            ->recordActions([
                ActionGroup::make([
                    ViewAction::make()->label('عرض'),
                    EditAction::make()->label('تعديل'),
                ]),
            ])

            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make()->label('حذف'),
                    ForceDeleteBulkAction::make()->label('حذف نهائي'),
                    RestoreBulkAction::make()->label('استرجاع'),
                ]),
            ]);
    }
}
