<?php

namespace App\Filament\Resources\ClientReceipts\Schemas;

use App\Models\ClientAccount;
use App\Models\ClientReceipt;
use App\Models\Invoice;
use App\Models\PaymentMethod;
use Coolsam\Flatpickr\Forms\Components\Flatpickr;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Components\Utilities\Set;
use Filament\Schemas\Schema;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Schema as DBSchema;

class ClientReceiptForm
{
    /* ===================== Helpers: Branch ===================== */

    protected static function currentBranchId(): int
    {
        try {
            if (function_exists('user_info')) {
                $b = user_info('branch_id');
                if ($b) {
                    return (int) $b;
                }
            }
        } catch (\Throwable) {
            // ignore
        }

        return (int) (user_info('branch_id') ?? 0);
    }

    /* ===================== Helpers: Integer Formatting ===================== */

    protected static function nfInt(int|string|null $v): string
    {
        if ($v === null || $v === '') {
            return '0';
        }

        if (is_string($v)) {
            $v = strtr($v, [
                '٠' => '0',
                '١' => '1',
                '٢' => '2',
                '٣' => '3',
                '٤' => '4',
                '٥' => '5',
                '٦' => '6',
                '٧' => '7',
                '٨' => '8',
                '٩' => '9',
                '۰' => '0',
                '۱' => '1',
                '۲' => '2',
                '۳' => '3',
                '۴' => '4',
                '۵' => '5',
                '۶' => '6',
                '۷' => '7',
                '۸' => '8',
                '۹' => '9',
            ]);

            $raw = str_replace([',', ' ', '.'], '', $v);
            if ($raw === '' || ! ctype_digit(ltrim($raw, '-'))) {
                return '0';
            }

            return number_format((int) $raw, 0, '.', ',');
        }

        return number_format((int) $v, 0, '.', ',');
    }

    protected static function toInt(null|string|int $v): int
    {
        if ($v === null || $v === '') {
            return 0;
        }
        if (is_int($v)) {
            return $v;
        }

        $v = (string) $v;
        $v = strtr($v, [
            '٠' => '0',
            '١' => '1',
            '٢' => '2',
            '٣' => '3',
            '٤' => '4',
            '٥' => '5',
            '٦' => '6',
            '٧' => '7',
            '٨' => '8',
            '٩' => '9',
            '۰' => '0',
            '۱' => '1',
            '۲' => '2',
            '۳' => '3',
            '۴' => '4',
            '۵' => '5',
            '۶' => '6',
            '۷' => '7',
            '۸' => '8',
            '۹' => '9',
        ]);

        $v = str_replace([',', ' ', '.'], '', $v);

        return (int) ($v === '' ? 0 : $v);
    }

    protected static function fmtDate(?string $v): ?string
    {
        if (! $v) {
            return null;
        }
        try {
            return Carbon::parse($v)->format('Y-m-d');
        } catch (\Throwable) {
            return (string) $v;
        }
    }

    /* ===================== Debt: From client_accounts (preferred) ===================== */

    protected static function clientDebtSummary(int $clientId, int $branchId): array
    {
        if (
            DBSchema::hasTable('client_accounts')
            && DBSchema::hasColumn('client_accounts', 'receivable_total')
            && DBSchema::hasColumn('client_accounts', 'receivable_paid')
            && DBSchema::hasColumn('client_accounts', 'receivable_remaining')
        ) {
            $acc = ClientAccount::query()
                ->where('client_id', $clientId)
                ->where('branch_id', $branchId)
                ->first();

            if ($acc) {
                $ti = (int) ($acc->receivable_total ?? 0);
                $tp = (int) ($acc->receivable_paid ?? 0);
                $td = (int) ($acc->receivable_remaining ?? max(0, $ti - $tp));

                return [
                    'total_invoices' => $ti,
                    'total_paid' => $tp,
                    'total_due' => $td,
                ];
            }
        }

        // احتياطي: من جدول invoices (حسب فرع/عميل)
        if (DBSchema::hasTable('invoices')) {
            $q = Invoice::query()
                ->where('client_id', $clientId)
                ->where('branch_id', $branchId);

            // إن تحب تستثني المسودات:
            // $q->where('status', '!=', 'draft');

            $ti = (int) $q->sum('grand_total');
            $tp = (int) $q->sum('paid_amount');
            $td = (int) max(0, $ti - $tp);

            return [
                'total_invoices' => $ti,
                'total_paid' => $tp,
                'total_due' => $td,
            ];
        }

        return ['total_invoices' => 0, 'total_paid' => 0, 'total_due' => 0];
    }

    /* ===================== Last Receipt (آخر دفعة مالية) ===================== */

    protected static function lastReceiptFor(int $clientId, int $branchId): ?array
    {
        if (! DBSchema::hasTable('client_receipts')) {
            return null;
        }

        $r = ClientReceipt::query()
            ->where('client_id', $clientId)
            ->where('branch_id', $branchId)
            ->where('status', 'issued')
            ->where('receipt_type', 'collection') // قبض من العميل = دفعة مالية
            ->orderByDesc('receipt_date')
            ->orderByDesc('id')
            ->first();

        if (! $r) {
            return null;
        }

        return [
            'receipt_number' => $r->receipt_number,
            'total_paid' => (int) ($r->total_paid ?? 0),
            'remaining_after' => (int) ($r->total_remaining ?? 0),
            'notes' => $r->payment_details ?: ($r->notes ?? null),
            'date' => self::fmtDate($r->receipt_date),
        ];
    }

    /* ===================== Form ===================== */

    public static function configure(Schema $schema): Schema
    {
        $jsFormatInt = "
            let val = this.value
                .replace(/[٠-٩]/g, d => '٠١٢٣٤٥٦٧٨٩'.indexOf(d))
                .replace(/[۰-۹]/g, d => '۰۱۲۳۴۵۶۷۸۹'.indexOf(d))
                .replace(/[,.\\s]/g, '');
            this.value = val ? val.replace(/\\B(?=(\\d{3})+(?!\\d))/g, ',') : '';
        ";

        return $schema->components([

            Section::make('📋 معلومات الوصل والتسديد')
                ->description('البيانات الأساسية للوصل، حالة ديون العميل وطريقة الدفع')
                ->schema([
                    // branch_id / receipt_type (مخفية)
                    Hidden::make('branch_id')
                        ->default(fn () => self::currentBranchId())
                        ->dehydrateStateUsing(fn ($state) => (int) ($state ?: self::currentBranchId())),

                    Hidden::make('receipt_type')
                        ->default('collection')
                        ->dehydrateStateUsing(fn ($state) => $state ?: 'collection'),

                    Grid::make(12)
                        ->schema([
                            Select::make('client_id')
                                ->label('العميل')
                                ->relationship('client', 'name')
                                ->searchable()
                                ->preload(false)
                                ->required()
                                ->live(debounce: 500)
                                ->columnSpan(4)
                                ->afterStateUpdated(function (?string $state, Set $set, Get $get) {

                                    foreach (
                                        [
                                            'total_invoices',
                                            'total_paid_sum',
                                            'total_due',
                                            'prev_receipt_number',
                                            'prev_receipt_amount',
                                            'prev_receipt_remaining',
                                            'prev_receipt_notes',
                                            'prev_receipt_date',
                                            'current_remaining',
                                            // حقول تخزين الوصل
                                            'total_amount',
                                            'total_remaining',
                                        ] as $f
                                    ) {
                                        $set($f, '0');
                                    }

                                    if (! $state) {
                                        return;
                                    }

                                    $clientId = (int) $state;
                                    $branchId = (int) ($get('branch_id') ?: self::currentBranchId());

                                    // ✅ مجاميع الديون من client_accounts (أو fallback invoices)
                                    $sum = self::clientDebtSummary($clientId, $branchId);

                                    $set('total_invoices', self::nfInt($sum['total_invoices']));
                                    $set('total_paid_sum', self::nfInt($sum['total_paid']));
                                    $set('total_due', self::nfInt($sum['total_due']));

                                    // نخلي الدين قبل الدفع كقيمة مرجعية (total_amount)
                                    $set('total_amount', self::nfInt($sum['total_due']));

                                    // ✅ آخر وصل (آخر دفعة مالية)
                                    $last = self::lastReceiptFor($clientId, $branchId);
                                    if ($last) {
                                        $set('prev_receipt_number', $last['receipt_number']);
                                        $set('prev_receipt_amount', self::nfInt($last['total_paid']));
                                        $set('prev_receipt_remaining', self::nfInt($last['remaining_after']));
                                        $set('prev_receipt_notes', $last['notes']);
                                        $set('prev_receipt_date', $last['date']);
                                    }
                                }),
                            Select::make('payment_method_id')
                                ->label('طريقة الدفع')
                                ->options(fn () => PaymentMethod::pluck('name', 'id')->all())
                                ->searchable()
                                ->preload()
                                ->columnSpan(4),

                            Flatpickr::make('receipt_date')
                                ->label('تاريخ الوصل')
                                ->allowInput()
                                ->altInput(true)
                                ->altFormat('Y-m-d')
                                ->format('Y-m-d')
                                ->default(now())
                                ->required()
                                ->columnSpan(4),
                        ]),

                    // الخزنة مخفية ومحددة تلقائيًا
                    Hidden::make('vault_id')
                        ->default(fn () => optional(function_exists('current_vault') ? current_vault() : null)->id)
                        ->afterStateHydrated(function (Set $set, $state) {
                            if (! $state && function_exists('current_vault')) {
                                $set('vault_id', optional(current_vault())->id);
                            }
                        }),

                    Grid::make(3)->schema([
                        TextInput::make('total_invoices')
                            ->label('💵 مجموع الفواتير')
                            ->suffix('دينار')
                            ->readOnly()->dehydrated(false)->default('0')
                            ->extraAttributes(['class' => 'text-lg font-bold'])
                            ->formatStateUsing(fn ($v) => self::nfInt($v)),

                        TextInput::make('total_paid_sum')
                            ->label('✅ مجموع المدفوع')
                            ->suffix('دينار')
                            ->readOnly()->dehydrated(false)->default('0')
                            ->extraAttributes(['class' => 'text-lg font-bold text-success-600'])
                            ->formatStateUsing(fn ($v) => self::nfInt($v)),

                        TextInput::make('total_due')
                            ->label('⚠️ الدين المتبقي')
                            ->suffix('دينار')
                            ->readOnly()->dehydrated(false)->default('0')
                            ->extraAttributes(['class' => 'text-lg font-bold text-danger-600'])
                            ->formatStateUsing(fn ($v) => self::nfInt($v)),
                    ]),
                ])
                ->columnSpanFull()
                ->collapsible(),

            Section::make('💰 الدفعة الحالية + آخر وصل')
                ->description('إدخال الدفعة الحالية وعرض آخر دفعة مالية (وصل)')
                ->schema([
                    Grid::make(2)->schema([

                        Section::make('✨ الدفعة الحالية')
                            ->description('المبلغ الذي سيتم تسديده الآن')
                            ->schema([

                                // نخزّن مرجع الدين قبل الدفع في total_amount (مهم للتاريخ)
                                Hidden::make('total_amount')
                                    ->default('0')
                                    ->dehydrateStateUsing(fn ($state, Get $get) => self::toInt($get('total_due'))),

                                TextInput::make('total_paid')
                                    ->label('المبلغ المدفوع*')
                                    ->suffix('دينار')
                                    ->default('0')
                                    ->required()
                                    ->live(debounce: 600)
                                    ->afterStateUpdated(function ($state, Set $set, Get $get) {
                                        $paid = self::toInt($state);
                                        $due = self::toInt($get('total_due'));

                                        $remaining = max(0, $due - $paid);

                                        // للعرض
                                        $set('current_remaining', self::nfInt($remaining));

                                        // للتخزين
                                        $set('total_remaining', self::nfInt($remaining));
                                    })
                                    ->dehydrateStateUsing(fn ($state) => self::toInt($state))
                                    ->formatStateUsing(fn ($v) => self::nfInt($v))
                                    ->extraInputAttributes(['onkeyup' => $jsFormatInt]),

                                // نخزّن المتبقي بعد الوصل في total_remaining
                                Hidden::make('total_remaining')
                                    ->default('0')
                                    ->dehydrateStateUsing(fn ($state) => self::toInt($state)),

                                TextInput::make('current_remaining')
                                    ->label('المتبقي بعد الدفعة')
                                    ->suffix('دينار')
                                    ->readOnly()
                                    ->dehydrated(false)
                                    ->default('0')
                                    ->extraAttributes(['class' => 'text-lg font-bold text-warning-600'])
                                    ->formatStateUsing(fn ($v) => self::nfInt($v)),

                                Textarea::make('payment_details')
                                    ->label('ملاحظات / تفاصيل الدفع')
                                    ->rows(3)
                                    ->placeholder('رقم تحويل / تفاصيل / ملاحظة...'),
                            ])
                            ->columnSpan(1),

                        Section::make('📌 آخر وصل (آخر دفعة مالية)')
                            ->description(function (Get $get) {
                                $d = $get('prev_receipt_date');
                                $n = $get('prev_receipt_number');
                                $txt = $d ? "آخر وصل بتاريخ {$d}" : 'آخر وصل';

                                return $n ? "{$txt} — رقم: {$n}" : $txt;
                            })
                            ->schema([

                                TextInput::make('prev_receipt_amount')
                                    ->label('المبلغ المدفوع')
                                    ->suffix('دينار')
                                    ->readOnly()
                                    ->dehydrated(false)
                                    ->default('0')
                                    ->extraAttributes(['class' => 'text-lg font-bold text-success-600'])
                                    ->formatStateUsing(fn ($v) => self::nfInt($v)),

                                TextInput::make('prev_receipt_remaining')
                                    ->label('المتبقي بعده')
                                    ->suffix('دينار')
                                    ->readOnly()
                                    ->dehydrated(false)
                                    ->default('0')
                                    ->extraAttributes(['class' => 'text-lg font-bold'])
                                    ->formatStateUsing(fn ($v) => self::nfInt($v)),

                                Hidden::make('prev_receipt_number')->dehydrated(false),
                                Hidden::make('prev_receipt_date')->dehydrated(false),

                                Textarea::make('prev_receipt_notes')
                                    ->label('ملاحظات الوصل')
                                    ->rows(3)
                                    ->readOnly()
                                    ->dehydrated(false)
                                    ->placeholder('—'),
                            ])
                            ->columnSpan(1),

                    ]),
                ])
                ->columnSpanFull()
                ->collapsible(),
        ]);
    }
}
