<?php

namespace App\Filament\Resources\ClientReceipts\Pages;

use App\Filament\Resources\ClientReceipts\ClientReceiptResource;
use App\Services\Accounting\ClientAccountingService;
use App\Services\Accounting\Receipts\DeleteClientReceiptService;
use App\Services\Accounting\Receipts\UpdateClientReceiptService;
use Filament\Actions\Action;
use Filament\Actions\ForceDeleteAction;
use Filament\Actions\RestoreAction;
use Filament\Actions\ViewAction;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\EditRecord;
use Illuminate\Database\Eloquent\Model;

class EditClientReceipt extends EditRecord
{
    protected static string $resource = ClientReceiptResource::class;

    /**
     * تحديث الوصل عبر خدمة محاسبية
     */
    protected function handleRecordUpdate(Model $record, array $data): Model
    {
        try {
            /** @var UpdateClientReceiptService $svc */
            $svc = app(UpdateClientReceiptService::class);

            /** @var Model $updated */
            $updated = $svc->execute($record, $data);

            // (اختياري) إعادة بناء balances للأستاذ بعد التحديث
            $acc = app(ClientAccountingService::class);
            if (method_exists($acc, 'rebuildClientLedgerBalances')) {
                $acc->rebuildClientLedgerBalances((int) $updated->client_id, (int) $updated->branch_id);
            }

            Notification::make()
                ->title('تم تحديث الوصل')
                ->body('تمت إعادة التوزيع وتحديث الفواتير ودفتر الأستاذ وحساب العميل.')
                ->success()
                ->send();

            return $updated;
        } catch (\Throwable $e) {
            report($e);

            Notification::make()
                ->title('تعذّر تحديث الوصل')
                ->body($e->getMessage() ?: 'حدث خطأ أثناء التحديث المحاسبي.')
                ->danger()
                ->send();

            throw $e;
        }
    }

    protected function getHeaderActions(): array
    {
        return [
            ViewAction::make(),

            // هذه تخص soft delete داخل Filament فقط
            ForceDeleteAction::make(),
            RestoreAction::make(),

            Action::make('deleteReceiptProperly')
                ->label('حذف مع تنظيف محاسبي')
                ->color('danger')
                ->icon('heroicon-o-trash')
                ->requiresConfirmation()
                ->modalHeading('تأكيد الحذف')
                ->modalDescription('سيتم حذف الوصل مع إزالة التخصيصات وتحديث الفواتير وحركة الخزنة ودفتر الأستاذ وحساب العميل.')
                ->action(function (): void {
                    try {
                        $record = $this->getRecord();

                        /** @var DeleteClientReceiptService $svc */
                        $svc = app(DeleteClientReceiptService::class);
                        $svc->execute($record);

                        // (اختياري) إعادة بناء balances بعد الحذف
                        $acc = app(ClientAccountingService::class);
                        if ($record?->client_id && $record?->branch_id && method_exists($acc, 'rebuildClientLedgerBalances')) {
                            $acc->rebuildClientLedgerBalances((int) $record->client_id, (int) $record->branch_id);
                        }

                        Notification::make()
                            ->title('تم الحذف')
                            ->body('تم تنظيف كافة الآثار المحاسبية للوصل بنجاح.')
                            ->success()
                            ->send();

                        $this->redirect(static::getResource()::getUrl('index'));
                    } catch (\Throwable $e) {
                        report($e);

                        Notification::make()
                            ->title('تعذّر الحذف')
                            ->body($e->getMessage() ?: 'حدث خطأ أثناء الحذف المحاسبي.')
                            ->danger()
                            ->send();
                    }
                }),
        ];
    }

    /**
     * بعد الحفظ نرجع لصفحة العرض
     */
    protected function getRedirectUrl(): string
    {
        return static::getResource()::getUrl('view', [
            'record' => $this->getRecord(),
        ]);
    }
}
