<?php

namespace App\Filament\Resources\ClientDiscounts\Tables;

use App\Models\ClientDiscount;
use Filament\Actions\ActionGroup;
use Filament\Actions\BulkAction;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TernaryFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class ClientDiscountsTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('client.name')
                    ->label('العميل')
                    ->searchable()
                    ->sortable()
                    ->weight('bold')
                    ->description(fn (ClientDiscount $record) => $record->client?->phone),

                TextColumn::make('discount_type')
                    ->label('النوع')
                    ->badge()
                    ->formatStateUsing(fn (string $state) => match ($state) {
                        'percentage' => 'نسبة مئوية',
                        'fixed' => 'مبلغ ثابت',
                        default => $state,
                    })
                    ->color(fn (string $state) => match ($state) {
                        'percentage' => 'success',
                        'fixed' => 'info',
                        default => 'gray',
                    }),

                TextColumn::make('discount_value')
                    ->label('قيمة الخصم')
                    ->formatStateUsing(function (ClientDiscount $record): string {
                        if ($record->discount_type === 'percentage') {
                            return $record->discount_value.'%';
                        }

                        return number_format($record->discount_value, 0, '', ',').' د.ع';
                    })
                    ->badge()
                    ->color('success'),

                IconColumn::make('is_active')
                    ->label('مفعّل')
                    ->boolean()
                    ->alignCenter()
                    ->sortable(),

                TextColumn::make('is_valid')
                    ->label('الحالة')
                    ->badge()
                    ->formatStateUsing(fn (ClientDiscount $record) => $record->is_valid ? 'ساري' : 'غير ساري')
                    ->color(fn (ClientDiscount $record) => $record->is_valid ? 'success' : 'danger')
                    ->icon(fn (ClientDiscount $record) => $record->is_valid ? 'heroicon-o-check-circle' : 'heroicon-o-x-circle'),

                TextColumn::make('starts_at')
                    ->label('يبدأ')
                    ->dateTime('Y-m-d')
                    ->sortable()
                    ->placeholder('فوري')
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('ends_at')
                    ->label('ينتهي')
                    ->dateTime('Y-m-d')
                    ->sortable()
                    ->placeholder('بدون نهاية')
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('createdBy.name')
                    ->label('أنشأه')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->dateTime('Y-m-d')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                SelectFilter::make('client_id')
                    ->label('العميل')
                    ->relationship('client', 'name')
                    ->searchable()
                    ->preload()
                    ->multiple(),

                SelectFilter::make('discount_type')
                    ->label('النوع')
                    ->options([
                        'percentage' => 'نسبة مئوية',
                        'fixed' => 'مبلغ ثابت',
                    ])
                    ->multiple(),

                TernaryFilter::make('is_active')
                    ->label('الحالة')
                    ->placeholder('الكل')
                    ->trueLabel('مفعّل')
                    ->falseLabel('معطّل'),

                Filter::make('valid')
                    ->label('ساري المفعول')
                    ->query(fn (Builder $query) => $query->valid()),

                Filter::make('expired')
                    ->label('منتهي')
                    ->query(fn (Builder $query) => $query
                        ->where('is_active', true)
                        ->whereNotNull('ends_at')
                        ->where('ends_at', '<', now())),
            ])
            ->recordActions([
                ActionGroup::make([
                    EditAction::make(),
                    DeleteAction::make()
                        ->requiresConfirmation(),
                ]),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make()
                        ->requiresConfirmation(),

                    BulkAction::make('activate')
                        ->label('تفعيل')
                        ->icon('heroicon-o-check-circle')
                        ->color('success')
                        ->requiresConfirmation()
                        ->action(fn ($records) => $records->each->update(['is_active' => true]))
                        ->deselectRecordsAfterCompletion(),

                    BulkAction::make('deactivate')
                        ->label('إلغاء التفعيل')
                        ->icon('heroicon-o-x-circle')
                        ->color('danger')
                        ->requiresConfirmation()
                        ->action(fn ($records) => $records->each->update(['is_active' => false]))
                        ->deselectRecordsAfterCompletion(),
                ]),
            ])
            ->defaultSort('created_at', 'desc')
            ->emptyStateHeading('لا توجد خصومات')
            ->emptyStateDescription('ابدأ بإضافة خصم للعملاء')
            ->emptyStateIcon('heroicon-o-receipt-percent')
            ->striped();
    }
}
