<?php

namespace App\Filament\Resources\Categories\Tables;

use App\Models\Category;
use Filament\Tables\Table;
use Filament\Actions\BulkAction;
use Filament\Actions\EditAction;
use Filament\Actions\ViewAction;
use Filament\Actions\ActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Tables\Filters\Filter;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TernaryFilter;
use Illuminate\Contracts\Database\Eloquent\Builder;

class CategoriesTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('name')
                    ->label('الاسم')
                    ->searchable()
                    ->sortable()
                    ->description(
                        fn(Category $record): ?string =>
                        $record->parent ? "تحت: {$record->parent->name}" : null
                    )
                    ->weight('bold'),

                TextColumn::make('parent.name')
                    ->label('التصنيف الأب')
                    ->searchable()
                    ->sortable()
                    ->placeholder('تصنيف رئيسي')
                    ->badge()
                    ->color('gray'),

                TextColumn::make('children_count')
                    ->label('التصنيفات الفرعية')
                    ->counts('children')
                    ->badge()
                    ->color('info')
                    ->alignCenter(),

                TextColumn::make('products_count')
                    ->label('عدد المنتجات')
                    ->counts('products')
                    ->badge()
                    ->color('success')
                    ->alignCenter(),

                IconColumn::make('is_active')
                    ->label('الحالة')
                    ->boolean()
                    ->trueIcon('heroicon-o-check-circle')
                    ->falseIcon('heroicon-o-x-circle')
                    ->trueColor('success')
                    ->falseColor('danger')
                    ->alignCenter(),

                TextColumn::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->dateTime('Y-m-d H:i')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('updated_at')
                    ->label('تاريخ التحديث')
                    ->dateTime('Y-m-d H:i')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                SelectFilter::make('parent_id')
                    ->label('التصنيف الأب')
                    ->options(
                        Category::query()
                            ->orderBy('name')
                            ->get()
                            ->unique('name')    // فلترة النتائج داخل المجموعة
                            ->pluck('name', 'id')
                    )
                    ->searchable()
                    ->multiple()
                    ->placeholder('الكل'),

                TernaryFilter::make('is_active')
                    ->label('الحالة')
                    ->searchable()
                    ->placeholder('الكل')
                    ->trueLabel('نشط')
                    ->falseLabel('متوقف'),

                Filter::make('main_categories')
                    ->label('التصنيفات الرئيسية فقط')
                    ->query(fn(Builder $query): Builder => $query->whereNull('parent_id'))
                    ->toggle(),

                Filter::make('has_products')
                    ->label('يحتوي على منتجات')
                    ->query(fn(Builder $query): Builder => $query->has('products'))
                    ->toggle(),

                Filter::make('has_children')
                    ->label('يحتوي على تصنيفات فرعية')
                    ->query(fn(Builder $query): Builder => $query->has('children'))
                    ->toggle(),
            ])
            ->recordActions([
                ActionGroup::make([
                    ViewAction::make(),
                    EditAction::make(),
                    DeleteAction::make()
                        ->requiresConfirmation()
                        ->modalHeading('حذف التصنيف')
                        ->modalDescription('هل أنت متأكد من حذف هذا التصنيف؟ سيتم حذف جميع التصنيفات الفرعية أيضاً.')
                        ->modalSubmitActionLabel('نعم، احذف'),
                ]),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make()
                        ->requiresConfirmation()
                        ->modalHeading('حذف التصنيفات المحددة')
                        ->modalDescription('هل أنت متأكد من حذف التصنيفات المحددة؟')
                        ->modalSubmitActionLabel('نعم، احذف'),

                    BulkAction::make('activate')
                        ->label('تفعيل')
                        ->icon('heroicon-o-check-circle')
                        ->color('success')
                        ->requiresConfirmation()
                        ->modalHeading('تفعيل التصنيفات')
                        ->modalDescription('سيتم تفعيل جميع التصنيفات المحددة')
                        ->modalSubmitActionLabel('نعم، فعّل')
                        ->action(fn($records) => $records->each->update(['is_active' => true]))
                        ->deselectRecordsAfterCompletion(),

                    BulkAction::make('deactivate')
                        ->label('إلغاء التفعيل')
                        ->icon('heroicon-o-x-circle')
                        ->color('danger')
                        ->requiresConfirmation()
                        ->modalHeading('إلغاء تفعيل التصنيفات')
                        ->modalDescription('سيتم إلغاء تفعيل جميع التصنيفات المحددة')
                        ->modalSubmitActionLabel('نعم، ألغِ التفعيل')
                        ->action(fn($records) => $records->each->update(['is_active' => false]))
                        ->deselectRecordsAfterCompletion(),
                ]),
            ]);
    }
}
