<?php

namespace App\Filament\Resources\Branches\Schemas;

use App\Models\Branch;
use Filament\Forms\Components\ColorPicker;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\TextInput;
use Filament\Schemas\Components\Group;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;

class BranchForm
{
    public static function configure(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make('بيانات الفرع')
                    ->columnSpanFull()
                    ->schema([
                        TextInput::make('name')
                            ->label('اسم الفرع')
                            ->required(),

                        // المحافظة
                        Select::make('city_id')
                            ->label('المحافظة')
                            ->relationship('city', 'name')
                            ->searchable()
                            ->preload()
                            ->native(false)
                            ->required(),

                        // الحالة
                        Select::make('status')
                            ->label('الحالة')
                            ->options(Branch::getStatuses())
                            ->required()
                            ->visibleOn('edit')
                            ->default(Branch::STATUS_ACTIVE),
                    ])
                    ->columns(3),

                /**
                 * كل ما يخص BranchSetting (ومنها الميديا) يجب أن يكون داخل relationship('settings')
                 * لأن BranchSetting هو HasMedia وليس Branch.
                 */
                Group::make()
                    ->relationship('settings') // ✅ هنا يصبح record هو BranchSetting
                    ->columnSpanFull()
                    ->schema([
                        Section::make('إعدادات الفرع')
                            ->schema([
                                TextInput::make('address')
                                    ->label('العنوان')
                                    ->maxLength(255),
                                // الهواتف (عندك phones array)
                                // إذا تريدها رقم واحد فقط: غيّرها إلى phone في جدول branches أو عدّل الموديل.
                                // هنا خليتها نص بسيط لتجنب تعقيد repeater بدون طلب صريح.
                                TextInput::make('phones.0')
                                    ->label('رقم الهاتف')
                                    ->tel()
                                    ->extraInputAttributes([
                                        'inputmode' => 'numeric',
                                        'class' => 'text-right',
                                    ])
                                    ->minLength(11)
                                    ->maxLength(11)
                                    ->rule('digits:11')
                                    ->dehydrateStateUsing(fn (?string $state) => $state ? preg_replace('/\D+/', '', $state) : null),

                                ColorPicker::make('theme_primary')
                                    ->label('اللون الأساسي للقاعة / الفاتورة')
                                    ->format('hex')
                                    ->required()
                                    ->helperText('اختر لونًا واحدًا فقط. سيتم توليد لوحة الألوان تلقائياً عند العرض.'),

                            ])
                            ->columns(3),

                        Section::make('الشعارات والملفات')
                            ->schema([
                                SpatieMediaLibraryFileUpload::make('logo_light')
                                    ->label('شعار فاتح')
                                    ->collection('logo_light')
                                    ->image()
                                    ->imageEditor()
                                    ->imageResizeMode('cover')
                                    ->imageCropAspectRatio('1:1')
                                    ->maxSize(2048)
                                    ->downloadable()
                                    ->openable(),

                                SpatieMediaLibraryFileUpload::make('logo_dark')
                                    ->label('شعار داكن')
                                    ->collection('logo_dark')
                                    ->image()
                                    ->imageEditor()
                                    ->imageResizeMode('cover')
                                    ->imageCropAspectRatio('1:1')
                                    ->maxSize(2048)
                                    ->downloadable()
                                    ->openable(),

                                SpatieMediaLibraryFileUpload::make('favicon')
                                    ->label('Favicon')
                                    ->collection('favicon')
                                    ->image()
                                    ->maxSize(512)
                                    ->downloadable()
                                    ->openable()
                                    ->helperText('PNG / ICO / SVG'),

                            ])
                            ->columns(3),

                    ]),
            ]);
    }
}
