<?php

namespace App\Filament\Resources\BranchSettings\Schemas;

use App\Models\City;
use Filament\Forms\Components\ColorPicker;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Tabs;
use Filament\Schemas\Components\Tabs\Tab;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Schema;

class BranchSettingForm
{
    public static function configure(Schema $schema): Schema
    {
        return $schema
            ->components([
                Tabs::make('settingsTabs')
                    ->tabs([
                        /* ======================= التبويب 1: التطبيق ======================= */
                        Tab::make('الاعدادات العامة')
                            ->icon('heroicon-o-computer-desktop')
                            ->schema([
                                // اسم الفرع (يتم حفظه في جدول الفروع)
                                TextInput::make('branch_name')
                                    ->label('اسم الفرع')
                                    ->required()
                                    ->maxLength(150)
                                    // نملأه من علاقة الفرع
                                    ->afterStateHydrated(function ($component, $record) {
                                        if ($component && $record && $record->branch) {
                                            $component->state($record->branch->name);
                                        }
                                    })
                                    // نتأكد أنه يُرحّل للـ form data
                                    ->dehydrated(true),

                                // المحافظة (Select من City؛ تُحفظ في Branch.city_id)
                                Select::make('city_id')
                                    ->label('المحافظة')
                                    ->options(fn () => City::query()->orderBy('name')->pluck('name', 'id'))
                                    ->searchable()
                                    ->preload()
                                    ->native(false)
                                    ->required()
                                    ->helperText('سيتم حفظ المحافظة على مستوى الفرع.')
                                    // نملأها من الفرع الحالي
                                    ->afterStateHydrated(function ($component, $record) {
                                        if ($component && $record && $record->branch) {
                                            $component->state($record->branch->city_id);
                                        }
                                    })
                                    // مهم: نريد تمريرها إلى mutateFormDataBeforeSave
                                    ->dehydrated(true),

                                // العنوان (يُحفظ في settings نفسها)
                                Textarea::make('address')
                                    ->label('العنوان')
                                    ->rows(1),

                                Grid::make(3)->schema([
                                    SpatieMediaLibraryFileUpload::make('logo_light')
                                        ->label('شعار الفرع (فاتح)')
                                        ->collection('logo_light')
                                        ->image()
                                        ->imageEditor()
                                        ->directory('branches/logos')
                                        ->disk('upload')
                                        ->preserveFilenames()
                                        ->openable()
                                        ->downloadable()
                                        ->maxSize(2048), // 2MB

                                    SpatieMediaLibraryFileUpload::make('logo_dark')
                                        ->label('شعار الفرع (مظلم)')
                                        ->collection('logo_dark')
                                        ->image()
                                        ->imageEditor()
                                        ->directory('branches/logos')
                                        ->disk('upload')
                                        ->preserveFilenames()
                                        ->openable()
                                        ->downloadable()
                                        ->maxSize(2048),

                                    SpatieMediaLibraryFileUpload::make('favicon')
                                        ->label('أيقونة الفرع (Favicon)')
                                        ->collection('favicon')
                                        ->image()
                                        ->imageEditor()
                                        ->imageCropAspectRatio('1:1')
                                        ->directory('branches/favicons')
                                        ->disk('upload')
                                        ->preserveFilenames()
                                        ->openable()
                                        ->downloadable()
                                        ->maxSize(1024), // 1MB
                                ])->columnSpanFull(),

                                // لون القالب
                                ColorPicker::make('theme_primary')
                                    ->label('لون القالب')
                                    ->format('hex')
                                    ->helperText('سيتم توليد لوحة الألوان تلقائيًا في الموديل.'),
                            ])
                            ->columns(3),
                        /* =================== التبويب 2: معلومات الاتصال =================== */
                        Tab::make('معلومات الاتصال')
                            ->icon('heroicon-o-phone')
                            ->schema([
                                // رقم الهاتف الرئيسي => phones.main
                                TextInput::make('phones.main')
                                    ->label('رقم الهاتف الرئيسي')
                                    ->tel()
                                    ->prefixIcon('heroicon-o-phone')
                                    ->mask('99999999999')
                                    ->rule('digits:11')
                                    ->dehydrateStateUsing(fn ($state) => $state ? preg_replace('/\D+/', '', $state) : null)
                                    ->helperText('يجب أن يتكون من 11 رقمًا (مثل 07700000000)'),

                                // رقم هاتف الدعم => phones.support
                                TextInput::make('phones.support')
                                    ->label('رقم هاتف الدعم')
                                    ->tel()
                                    ->prefixIcon('heroicon-o-phone')
                                    ->mask('99999999999')
                                    ->rule('digits:11')
                                    ->dehydrateStateUsing(fn ($state) => $state ? preg_replace('/\D+/', '', $state) : null)
                                    ->helperText('أدخل رقم دعم يتكون من 11 رقمًا بالضبط'),

                                // البريد للدعم => socials.support_email (هنا فقط)
                                TextInput::make('socials.support_email')
                                    ->label('البريد الإلكتروني للدعم')
                                    ->email()
                                    ->prefixIcon('heroicon-o-envelope'),
                            ])
                            ->columns(3),

                        /* ================== التبويب 3: الشبكات الاجتماعية ================== */
                        Tab::make('الشبكات الاجتماعية')
                            ->icon('heroicon-o-heart')
                            ->schema([
                                // ✅ الروابط تُحفظ داخل socials.links (نخزن اليوزر فقط)
                                Repeater::make('socials.links')
                                    ->label('حسابات التواصل (اسم الحساب فقط)')
                                    ->schema([
                                        Select::make('platform')
                                            ->label('المنصة')
                                            ->options([
                                                'facebook' => 'Facebook',
                                                'instagram' => 'Instagram',
                                                'whatsapp' => 'WhatsApp',
                                                'tiktok' => 'TikTok',
                                                'x' => 'X (Twitter)',
                                                'youtube' => 'YouTube',
                                            ])
                                            ->required()
                                            ->native(false)
                                            ->columnSpan(2),

                                        TextInput::make('handle')
                                            ->label('اسم الحساب')
                                            ->placeholder('mystore أو 07700000000 للواتساب')
                                            ->required()
                                            ->columnSpan(4)
                                            ->helperText(
                                                fn (Get $get) => $get('platform') === 'whatsapp'
                                                    ? 'أدخل رقم الهاتف فقط، بدون + وبدون مسافات.'
                                                    : 'أدخل اليوزر بدون @ وبدون رابط.'
                                            )
                                            ->dehydrateStateUsing(function ($state, Get $get) {
                                                $state = ltrim(trim((string) $state), '@');
                                                if ($get('platform') === 'whatsapp') {
                                                    $state = preg_replace('/\D+/', '', $state);
                                                }

                                                return $state;
                                            })
                                            ->rule(
                                                fn (Get $get) => $get('platform') === 'whatsapp'
                                                    ? ['regex:/^[0-9]{7,15}$/']
                                                    : ['regex:/^[A-Za-z0-9._-]{2,30}$/']
                                            ),
                                    ])
                                    ->columns(6)
                                    ->defaultItems(0)
                                    ->addActionLabel('إضافة حساب جديد')
                                    ->reorderable()
                                    ->collapsible()
                                    // تنظيف أي مفاتيح زائدة عند الحفظ
                                    ->mutateDehydratedStateUsing(fn ($state) => collect($state)->map(
                                        fn ($row) => [
                                            'platform' => $row['platform'] ?? null,
                                            'handle' => $row['handle'] ?? null,
                                        ]
                                    )->all())
                                    ->itemLabel(fn (array $state): ?string => match ($state['platform'] ?? null) {
                                        'facebook' => 'Facebook',
                                        'instagram' => 'Instagram',
                                        'whatsapp' => 'WhatsApp',
                                        'tiktok' => 'TikTok',
                                        'x' => 'X (Twitter)',
                                        'youtube' => 'YouTube',
                                        default => 'حساب جديد',
                                    }),
                            ])
                            ->columns(1),

                        /* =================== التبويب 4: المخزن والعيارات =================== */
                    ])
                    ->persistTabInQueryString()
                    ->columnSpanFull(),
            ]);
    }
}
