<?php

namespace App\Filament\Resources\Attributes;

use UnitEnum;
use BackedEnum;
use App\Models\Attribute;
use App\Support\NavGroups;
use Filament\Tables\Table;
use Filament\Schemas\Schema;
use Illuminate\Validation\Rule;
use Filament\Actions\EditAction;
use Filament\Resources\Resource;
use Filament\Actions\ActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Support\Icons\Heroicon;
use Filament\Actions\BulkActionGroup;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Toggle;
use Filament\Actions\DeleteBulkAction;
use Filament\Forms\Components\Repeater;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Forms\Components\TextInput;
use Filament\Schemas\Components\Section;
use Filament\Tables\Filters\SelectFilter;
use Filament\Forms\Components\ColorPicker;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Forms\Components\Repeater\TableColumn;
use App\Filament\Resources\Attributes\Pages\ManageAttributes;

class AttributeResource extends Resource
{
    protected static ?string $model = Attribute::class;

    protected static string|BackedEnum|null $navigationIcon = Heroicon::OutlinedRectangleStack;
    protected static ?string $navigationLabel               = 'الخصائص';
    protected static ?string $modelLabel                    = 'خاصية';
    protected static ?string $pluralModelLabel              = 'الخصائص';
    protected static string|UnitEnum|null $navigationGroup  = NavGroups::PRODUCTS;
    protected static ?int $navigationSort                   = 64;
    protected static ?string $recordTitleAttribute          = 'name';

    public static function form(Schema $schema): Schema
    {
        return $schema
            ->columns(3)
            ->components([
                // ================== القسم الأول: المعلومات الأساسية ==================
                Section::make('المعلومات الأساسية')
                    ->icon('heroicon-o-information-circle')
                    ->description('البيانات الرئيسية للخاصية')
                    ->schema([
                        TextInput::make('display_name')
                            ->label('الاسم الظاهر')
                            ->placeholder('مثال: اللون، الحجم، المادة')
                            ->required()
                            ->maxLength(100)
                            ->prefixIcon('heroicon-o-tag'),

                        TextInput::make('name')
                            ->label('الاسم البرمجي (Slug)')
                            ->placeholder('مثال: color, size, material')
                            ->required()
                            ->unique(ignoreRecord: true)
                            ->maxLength(100)
                            ->prefixIcon('heroicon-o-code-bracket'),

                        Select::make('type')
                            ->label('نوع الخاصية')
                            ->options([
                                'text'  => '📝 نص',
                                'color' => '🎨 لون',
                            ])
                            ->default('text')
                            ->required()
                            ->live()
                            ->prefixIcon('heroicon-o-squares-2x2'),

                        TextInput::make('sort_order')
                            ->numeric()
                            ->label('ترتيب العرض')
                            ->default(function () {
                                $last = Attribute::max('sort_order');
                                return $last ? $last + 1 : 1;
                            })
                            ->minValue(0)
                            ->suffixIcon('heroicon-o-bars-3'),

                        Toggle::make('is_active')
                            ->label('تفعيل الخاصية')
                            ->inline(false)
                            ->visibleOn('edit')
                            ->default(true)
                            ->helperText('تعطيل الخاصية يخفيها من النظام'),
                    ])
                    ->collapsible()
                    ->columns(2)
                    ->columnSpanFull(),

                // ================== القسم الثاني: قيَم الخاصية (تكرار جدولي مدمج) ==================
                Section::make('القيَم')
                    ->schema([
                        Repeater::make('values')
                            ->relationship('values')
                            ->label('قيَم الخاصية')
                            ->orderColumn('sort_order')
                            ->defaultItems(1)
                            ->collapsible()
                            ->compact()

                            // هنا التعديل المهم: تغيير الأعمدة حسب العملية (إنشاء / تعديل)
                            ->columns(fn(Get $get, string $operation) => $operation === 'create' ? 2 : 3)
                            // لو تفضّل الـ Enum:
                            // ->columns(fn (Get $get, string $operation) => $operation === Operation::Create ? 2 : 3)

                            ->columnSpanFull()
                            ->schema([
                                TextInput::make('display_value')
                                    ->label('القيمة')
                                    ->required()
                                    ->maxLength(100),

                                TextInput::make('value')
                                    ->label('القيمة البرمجية')
                                    ->visible(false)
                                    ->maxLength(100),

                                ColorPicker::make('color_code')
                                    ->label('اللون')
                                    ->visible(fn(Get $get) => $get('../../type') === 'color')
                                    ->required(fn(Get $get) => $get('../../type') === 'color'),

                                Toggle::make('is_active')
                                    ->label('الحالة')
                                    ->inline(false)
                                    ->visibleOn('edit') // يبقى ظاهر فقط في وضع التعديل
                                    ->default(true),
                            ])

                    ])
                    ->collapsible()
                    ->collapsed(false)
                    ->columnSpanFull()
                    ->label('قيَم الخاصية'),
            ]);
    }


    public static function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('name')
            ->columns([
                TextColumn::make('name')
                    ->label('الاسم')
                    ->searchable(),
                TextColumn::make('display_name')
                    ->label('الاسم البرمجي')
                    ->searchable(),
                TextColumn::make('type')
                    ->label('النوع')
                    ->formatStateUsing(fn(string $state): string => match ($state) {
                        'text' => 'نص',
                        'color' => 'لون',
                    }),
                TextColumn::make('sort_order')
                    ->label('ترتيب')
                    ->badge()
                    ->color('primary')
                    ->sortable(),
                IconColumn::make('is_active')
                    ->boolean()
                    ->color(fn(bool $state): string => $state ? 'success' : 'danger')
                    ->label('الحالة'),
                TextColumn::make('created_at')
                    ->label('تم الإنشاء في')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('updated_at')
                    ->label('تم التحديث في')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                SelectFilter::make('type')
                    ->label('النوع')
                    ->placeholder('اختر النوع')
                    ->searchable()
                    ->preload()
                    ->options(['text' => 'نص', 'color' => 'لون']),
                SelectFilter::make('is_active')
                    ->label('الحالة')
                    ->placeholder('اختر الحالة')
                    ->searchable()
                    ->preload()
                    ->options([true => 'فعال', false => 'متوقف']),
            ])
            ->recordActions([
                ActionGroup::make([
                    EditAction::make(),
                    DeleteAction::make(),
                ]),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => ManageAttributes::route('/'),
        ];
    }
}
