<?php

namespace App\Filament\Resources\Attendances\Tables;

use Filament\Actions\ActionGroup;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TernaryFilter;
use Filament\Tables\Table;
use Illuminate\Support\Carbon;

class AttendancesTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('employee.name')
                    ->label('الموظف')
                    ->sortable()
                    ->searchable()
                    ->weight('bold')
                    ->description(fn($record) => $record->employee?->job_title),
                TextColumn::make('check_in')
                    ->label('التاريخ')
                    ->date('d M Y')
                    ->sortable(),
                TextColumn::make('check_in_time')
                    ->label('وقت الدخول')
                    ->state(fn($record) => $record->check_in?->format('h:i A'))
                    ->color(fn($record) => ($record->late_minutes ?? 0) > 0 ? 'warning' : null)
                    ->sortable(),
                TextColumn::make('check_out_time')
                    ->label('وقت الخروج')
                    ->state(fn($record) => $record->check_out?->format('h:i A'))
                    ->sortable(),
                TextColumn::make('branch.name')
                    ->label('الموقع')
                    ->badge()
                    ->sortable(),
                TextColumn::make('worked_minutes')
                    ->label('إجمالي العمل')
                    ->state(fn($record) => self::formatMinutes($record->worked_minutes))
                    ->alignEnd(),
                IconColumn::make('status')
                    ->label('الحالة')
                    ->icon(fn(string $state) => match ($state) {
                        'confirmed' => 'heroicon-o-check-circle',
                        'pending' => 'heroicon-o-clock',
                        'rejected' => 'heroicon-o-x-circle',
                        default => 'heroicon-o-question-mark-circle',
                    })
                    ->color(fn(string $state) => match ($state) {
                        'confirmed' => 'success',
                        'pending' => 'warning',
                        'rejected' => 'danger',
                        default => 'gray',
                    }),
            ])
            ->defaultSort('check_in', 'desc')
            ->filters([
                Filter::make('month')
                    ->label('الشهر')
                    ->form([
                        \Filament\Forms\Components\DatePicker::make('from')->label('من'),
                        \Filament\Forms\Components\DatePicker::make('to')->label('إلى'),
                    ])
                    ->query(function ($query, array $data) {
                        $from = $data['from'] ?? null;
                        $to = $data['to'] ?? null;
                        return $query
                            ->when($from, fn($q) => $q->whereDate('check_in', '>=', $from))
                            ->when($to, fn($q) => $q->whereDate('check_in', '<=', $to));
                    })
                    ->default([
                        'from' => Carbon::now()->startOfMonth(),
                        'to' => Carbon::now()->endOfMonth(),
                    ]),
                SelectFilter::make('branch_id')
                    ->label('الفرع')
                    ->relationship('branch', 'name')
                    ->visible(fn() => auth()->user()?->hasRole('admin')),
                SelectFilter::make('employee_id')
                    ->label('الموظف')
                    ->relationship('employee', 'name')
                    ->searchable(),
                TernaryFilter::make('overtime_approved')
                    ->label('موافقة إضافي'),
                SelectFilter::make('status')
                    ->label('الحالة')
                    ->options([
                        'pending' => 'قيد المراجعة',
                        'confirmed' => 'مؤكد',
                        'rejected' => 'مرفوض',
                    ]),
                SelectFilter::make('source')
                    ->label('المصدر')
                    ->options([
                        'manual' => 'يدوي',
                        'device' => 'جهاز',
                        'import' => 'استيراد',
                    ]),
            ])
            ->recordActions([
                ActionGroup::make([
                    EditAction::make(),
                    DeleteAction::make(),
                ]),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ]);
    }

    protected static function formatMinutes(?int $minutes): string
    {
        if ($minutes === null) {
            return '--';
        }

        $hours = intdiv($minutes, 60);
        $mins = $minutes % 60;

        return sprintf('%02dh %02dm', $hours, $mins);
    }
}
