<?php

namespace App\Filament\Pages;

use UnitEnum;
use BackedEnum;
use Filament\Tables;
use App\Models\Employee;
use Filament\Pages\Page;
use App\Models\Attendance;
use App\Support\NavGroups;
use Filament\Tables\Table;
use Filament\Actions\Action;
use Illuminate\Support\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Filament\Support\Icons\Heroicon;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\ViewColumn;
use Filament\Tables\Contracts\HasTable;
use Filament\Notifications\Notification;
use Filament\Tables\Concerns\InteractsWithTable;

class AttendanceGrid extends Page implements HasTable
{
    use InteractsWithTable;

    protected string $view = 'filament.pages.attendance-grid';
    protected static string $routePath = '/attendance-grid';
    protected static bool $shouldRegisterNavigation = false;

    protected static string|BackedEnum|null $navigationIcon = Heroicon::OutlinedRectangleStack;
    protected static string|UnitEnum|null $navigationGroup = NavGroups::MEMBERS;
    protected static ?string $navigationLabel = 'الحضور الشهري';
    protected static ?string $title = 'الحضور والانصراف';
    protected static ?int $navigationSort = 14;

    public int $year;
    public int $month;

    /** @var array<int,bool> */
    public array $days = [];

    /** @var array<int,array<int,bool>> employee_id => [day => present] */
    public array $matrix = [];

    private const MONTH_NAMES_AR = [
        1  => 'يناير',
        2  => 'فبراير',
        3  => 'مارس',
        4  => 'أبريل',
        5  => 'مايو',
        6  => 'يونيو',
        7  => 'يوليو',
        8  => 'أغسطس',
        9  => 'سبتمبر',
        10 => 'أكتوبر',
        11 => 'نوفمبر',
        12 => 'ديسمبر',
    ];

    public function mount(): void
    {
        $now = now();
        $this->year = (int) $now->year;
        $this->month = (int) $now->month;

        $this->loadMonthState();
    }

    public function goToMonth(int $year, int $month): void
    {
        $this->year = $year;
        $this->month = $month;

        $this->loadMonthState();
        $this->resetTable(); // مهم: لإعادة بناء الأعمدة/الصفوف
    }

    public function previousMonth(): void
    {
        $date = Carbon::create($this->year, $this->month, 1)->subMonth();
        $this->goToMonth((int) $date->year, (int) $date->month);
    }

    public function nextMonth(): void
    {
        $date = Carbon::create($this->year, $this->month, 1)->addMonth();
        $this->goToMonth((int) $date->year, (int) $date->month);
    }

    public function save(): void
    {
        $base = Carbon::create($this->year, $this->month, 1);

        $existing = Attendance::query()
            ->whereYear('check_in', $this->year)
            ->whereMonth('check_in', $this->month)
            ->get()
            ->groupBy(fn(Attendance $att) => $att->employee_id . '-' . $att->check_in->day);

        DB::transaction(function () use ($base, $existing) {
            foreach ($this->matrix as $employeeId => $days) {
                foreach ($days as $day => $present) {
                    $key = $employeeId . '-' . $day;
                    $current = $existing->get($key)?->first();

                    if ($present && ! $current) {
                        $checkIn = $base->copy()->day((int) $day)->setTime(9, 0);

                        Attendance::create([
                            'branch_id' => $this->getEmployeeBranch((int) $employeeId),
                            'employee_id' => (int) $employeeId,
                            'work_schedule_id' => null,
                            'check_in' => $checkIn,
                            'check_out' => null,
                            'worked_minutes' => null,
                            'late_minutes' => 0,
                            'overtime_minutes' => 0,
                            'overtime_approved' => false,
                            'source' => 'manual',
                            'status' => 'confirmed',
                            'notes' => null,
                        ]);
                    }

                    if (! $present && $current) {
                        $current->delete();
                    }
                }
            }
        });

        Notification::make()->success()->title('تم حفظ الحضور بنجاح.')->send();
    }

    protected function loadMonthState(): void
    {
        $date = Carbon::create($this->year, $this->month, 1);
        $this->days = range(1, $date->daysInMonth);

        // جلب حضور الشهر مرة واحدة (مهم للأداء)
        $attendances = Attendance::query()
            ->select('employee_id', 'check_in')
            ->whereYear('check_in', $this->year)
            ->whereMonth('check_in', $this->month)
            ->get()
            ->groupBy('employee_id')
            ->map(fn(Collection $rows) => $rows->groupBy(fn($row) => (int) $row->check_in->day));

        // جهّز مصفوفة matrix للموظفين الذين ضمن الاستعلام الأساسي (Active + branch scope)
        $employeeIds = $this->baseEmployeesQuery()->pluck('id');

        $matrix = [];
        foreach ($employeeIds as $empId) {
            foreach ($this->days as $day) {
                $matrix[$empId][$day] = isset($attendances[$empId]) && $attendances[$empId]->has($day);
            }
        }

        $this->matrix = $matrix;
    }

    protected function baseEmployeesQuery()
    {
        $query = Employee::query()
            ->where('is_active', true);

        if (! user_info()->hasRole('admin') && user_info('branch_id')) {
            $query->where('branch_id', user_info('branch_id'));
        }

        return $query;
    }

    protected function getEmployeeBranch(int $employeeId): int
    {
        return Employee::query()->whereKey($employeeId)->value('branch_id') ?? (int) (user_info('branch_id') ?? 0);
    }

    public function getCurrentLabelProperty(): string
    {
        $name = self::MONTH_NAMES_AR[$this->month] ?? sprintf('%02d', $this->month);
        return "{$name} {$this->year}";
    }

    public function table(Table $table): Table
    {
        // أعمدة الأيام (Dynamic ViewColumns)
        $dayColumns = [];
        foreach ($this->days as $day) {
            $label = str_pad((string) $day, 2, '0', STR_PAD_LEFT);

            $dayColumns[] = ViewColumn::make("day_{$day}")
                ->label($label)
                ->view('filament.tables.columns.attendance-day')
                ->viewData([
                    'day' => $day,
                ])
                ->alignCenter()
                ->extraHeaderAttributes([
                    'class' => 'whitespace-nowrap',
                ]);
        }

        return $table
            ->query($this->baseEmployeesQuery()->orderBy('name'))
            ->heading('جدول الحضور الشهري - ' . $this->currentLabel) // ← العنوان هنا
            ->description('تسجيل وإدارة حضور الموظفين لشهر ' . $this->currentLabel) // ← يمكنك إضافة وصف اختياري
            ->searchable()
            ->columns([
                TextColumn::make('name')
                    ->label('الموظف')
                    ->searchable()
                    ->sortable()
                    ->wrap()
                    ->extraAttributes([
                        'class' => 'font-semibold',
                    ]),
                ...$dayColumns,
            ])
            ->paginated([10, 25, 50])
            ->striped()
            ->headerActions([
                Action::make('prev')
                    ->label('السابق')
                    ->icon('heroicon-o-chevron-right')
                    ->color('gray')
                    ->action(fn() => $this->previousMonth()),

                Action::make('current')
                    ->label($this->currentLabel)
                    ->icon('heroicon-o-calendar')
                    ->color('primary')
                    ->disabled(),

                Action::make('next')
                    ->label('التالي')
                    ->icon('heroicon-o-chevron-left')
                    ->color('gray')
                    ->action(fn() => $this->nextMonth()),

                Action::make('save')
                    ->label('حفظ الحضور')
                    ->icon('heroicon-o-check-badge')
                    ->color('primary')
                    ->action(fn() => $this->save()),
            ])
            ->contentGrid([
                'md' => 1,
            ])
            ->emptyStateHeading('لا توجد بيانات موظفين');
    }
}
