<?php

namespace App\Filament\Pages;

use App\Models\User;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Notifications\Notification;
use Filament\Pages\Page;
use Filament\Schemas\Components\Tabs;
use Filament\Schemas\Components\Tabs\Tab;
use Filament\Schemas\Concerns\InteractsWithSchemas;
use Filament\Schemas\Contracts\HasSchemas;
use Filament\Schemas\Schema;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Illuminate\Validation\Rules\Password;

class AccountSettings extends Page implements HasSchemas
{
    use InteractsWithSchemas;

    protected static ?string $title = 'الملف الشخصي';

    protected static bool $shouldRegisterNavigation = false;

    protected string $view = 'filament.pages.account-settings';

    public array $data = [];

    /**
     * ✅ مهم جداً مع SpatieMediaLibraryFileUpload داخل Page:
     * يربط الفورم بموديل المستخدم لكي تُحفظ الـ media عبر saveRelationships().
     */
    protected function getFormModel(): User
    {
        /** @var User $user */
        $user = auth('web')->user();

        // ✅ تأكد من تحميل العلاقة media
        $user->loadMissing('media');

        return $user;
    }

    public function mount(): void
    {
        /** @var User $user */
        $user = $this->getFormModel();

        // ✅ لا تملأ avatar هنا، SpatieMediaLibraryFileUpload يقرأها من الموديل تلقائياً
        $this->form->fill([
            'name' => (string) $user->name,
            'username' => (string) $user->username,
            'email' => (string) $user->email,
            'phone' => (string) ($user->phone ?? ''),
            'address' => (string) ($user->address ?? ''),

            'password' => null,
            'password_confirmation' => null,

            'ui_preferences' => (array) ($user->ui_preferences ?? []),
        ]);
    }

    public function form(Schema $schema): Schema
    {
        $userId = (int) auth('web')->id();

        return $schema
            ->statePath('data')
            ->components([
                Tabs::make('account_tabs')
                    ->persistTabInQueryString()
                    ->tabs([
                        Tab::make('الملف الشخصي')
                            ->icon('heroicon-m-user')
                            ->schema([
                                SpatieMediaLibraryFileUpload::make('avatar')
                                    ->label('الصورة الشخصية')
                                    ->collection('avatar')
                                    ->disk('upload')
                                    ->image()
                                    ->avatar()
                                    ->imageEditor()
                                    ->openable()
                                    ->downloadable()
                                    ->maxFiles(1)
                                    ->columnSpanFull(),

                                TextInput::make('name')
                                    ->label('الاسم')
                                    ->required()
                                    ->maxLength(255)
                                    ->columnSpan(6),

                                TextInput::make('username')
                                    ->label('اسم المستخدم')
                                    ->required()
                                    ->maxLength(255)
                                    ->rule(Rule::unique('users', 'username')->ignore($userId))
                                    ->columnSpan(6),

                                TextInput::make('email')
                                    ->label('البريد الإلكتروني')
                                    ->email()
                                    ->required()
                                    ->maxLength(255)
                                    ->rule(Rule::unique('users', 'email')->ignore($userId))
                                    ->columnSpan(6),

                                TextInput::make('phone')
                                    ->label('رقم الهاتف (العراق)')
                                    ->tel()
                                    ->helperText('يجب أن يكون 11 رقم ويبدأ بـ 07')
                                    ->columnSpan(6)
                                    ->dehydrateStateUsing(function ($state) {
                                        $digits = preg_replace('/\D+/', '', (string) $state);

                                        return substr($digits ?? '', 0, 11);
                                    })
                                    ->rules([
                                        'nullable',
                                        'digits:11',
                                        'regex:/^07\d{9}$/',
                                    ])
                                    ->extraInputAttributes([
                                        'inputmode' => 'numeric',
                                        'dir' => 'ltr',
                                        'maxlength' => 11,
                                        'x-data' => '{}',
                                        'x-on:input' => <<<'JS'
                                            let v = $el.value.replace(/\D+/g, '');
                                            if (v.length > 11) v = v.slice(0, 11);
                                            $el.value = v;
                                        JS,
                                    ]),

                                Textarea::make('address')
                                    ->label('العنوان')
                                    ->rows(3)
                                    ->columnSpan(12),
                            ])
                            ->columns(12),

                        Tab::make('الأمان')
                            ->icon('heroicon-m-shield-check')
                            ->schema([
                                TextInput::make('password')
                                    ->label('كلمة المرور الجديدة')
                                    ->password()
                                    ->rule(Password::default())
                                    ->confirmed()
                                    ->dehydrateStateUsing(fn ($state) => filled($state) ? (string) $state : null)
                                    ->dehydrated(fn ($state) => filled($state))
                                    ->columnSpan(6),

                                TextInput::make('password_confirmation')
                                    ->label('تأكيد كلمة المرور')
                                    ->password()
                                    ->dehydrated(false)
                                    ->columnSpan(6),
                            ])
                            ->columns(12),
                    ]),
            ]);
    }

    public function save(): void
    {
        /** @var User $user */
        $user = auth('web')->user();
        $state = (array) $this->form->getState();

        // ✅ بيانات أساسية
        $user->name = (string) ($state['name'] ?? $user->name);
        $user->username = (string) ($state['username'] ?? $user->username);
        $user->email = (string) ($state['email'] ?? $user->email);
        $user->phone = (string) ($state['phone'] ?? '');
        $user->address = (string) ($state['address'] ?? '');

        // ✅ كلمة المرور (إن وُجدت)
        if (! empty($state['password'])) {
            $user->password = Hash::make((string) $state['password']);
        }

        $user->ui_preferences = (array) ($state['ui_preferences'] ?? []);
        $user->save();

        /**
         * ✅ الأهم: حفظ العلاقات المرتبطة بالفورم
         * (ومنها SpatieMediaLibraryFileUpload) داخل Page.
         */
        $this->form->model($user)->saveRelationships();

        // ✅ تحديث المستخدم وتحميل العلاقات
        $user->refresh();
        $user->load('media');

        // ✅ تحديث الجلسة
        auth('web')->setUser($user);

        Notification::make()
            ->title('تم حفظ التغييرات بنجاح')
            ->success()
            ->send();

        // ✅ الحل: إعادة تحميل الصفحة بالكامل لعرض الصورة
        $this->redirect(request()->header('Referer') ?: static::getUrl(), navigate: false);
    }
}
