<?php

namespace App\Enums;

/**
 * ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
 * حالات الشحنة
 * ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
 */
enum ShipmentStatus: string
{
    case Preparing = 'preparing';       // قيد التجهيز
    case Picked = 'picked';          // تم الاستلام من المندوب
    case InTransit = 'in_transit';      // في الطريق
    case OutForDelivery = 'out_for_delivery'; // خرج للتوصيل
    case Delivered = 'delivered';       // تم التوصيل
    case Returned = 'returned';        // مرتجع
    case Failed = 'failed';          // فشل التوصيل

    public function label(): string
    {
        return match ($this) {
            self::Preparing => 'قيد التجهيز',
            self::Picked => 'تم الاستلام',
            self::InTransit => 'في الطريق',
            self::OutForDelivery => 'خرج للتوصيل',
            self::Delivered => 'تم التوصيل',
            self::Returned => 'مرتجع',
            self::Failed => 'فشل التوصيل',
        };
    }

    public function color(): string
    {
        return match ($this) {
            self::Preparing => 'gray',
            self::Picked => 'info',
            self::InTransit => 'warning',
            self::OutForDelivery => 'primary',
            self::Delivered => 'success',
            self::Returned => 'danger',
            self::Failed => 'danger',
        };
    }

    public function icon(): string
    {
        return match ($this) {
            self::Preparing => 'heroicon-o-cube',
            self::Picked => 'heroicon-o-hand-raised',
            self::InTransit => 'heroicon-o-truck',
            self::OutForDelivery => 'heroicon-o-map-pin',
            self::Delivered => 'heroicon-o-check-circle',
            self::Returned => 'heroicon-o-arrow-uturn-left',
            self::Failed => 'heroicon-o-x-circle',
        };
    }

    public static function options(): array
    {
        return collect(self::cases())
            ->mapWithKeys(fn (self $status) => [$status->value => $status->label()])
            ->all();
    }

    /**
     * هل الشحنة في حالة نهائية؟
     */
    public function isFinal(): bool
    {
        return in_array($this, [
            self::Delivered,
            self::Returned,
            self::Failed,
        ]);
    }

    /**
     * هل الشحنة نشطة (قيد التوصيل)؟
     */
    public function isActive(): bool
    {
        return in_array($this, [
            self::Preparing,
            self::Picked,
            self::InTransit,
            self::OutForDelivery,
        ]);
    }
}
