<?php

namespace App\Enums;

use Filament\Support\Contracts\HasColor;
use Filament\Support\Contracts\HasIcon;
use Filament\Support\Contracts\HasLabel;

enum RoleEnum: string implements HasColor, HasIcon, HasLabel
{
    case admin = 'admin';
    case accounting_manager = 'accounting_manager';
    case branch_manager = 'branch_manager';
    case sales = 'sales';
    case accountant = 'accountant';
    case warehouse_keeper = 'warehouse_keeper';
    case driver = 'driver';
    case planning = 'planning';

    private const LABELS = [
        'admin' => 'المدير العام',
        'accounting_manager' => 'مدير المحاسبة',
        'branch_manager' => 'مدير الفرع',
        'sales' => 'قسم المبيعات',
        'accountant' => 'محاسب',
        'warehouse_keeper' => 'أمين المخزن',
        'driver' => 'سائق',
        'planning' => 'قسم التخطيط',
    ];

    private const COLORS = [
        'admin' => 'primary',
        'accounting_manager' => 'success',
        'branch_manager' => 'warning',
        'sales' => 'info',
        'accountant' => 'secondary',
        'warehouse_keeper' => 'gray',
        'driver' => 'danger',
        'planning' => 'primary',
    ];

    private const ICONS = [
        'admin' => 'heroicon-o-shield-check',
        'accounting_manager' => 'heroicon-o-banknotes',
        'branch_manager' => 'heroicon-o-building-office-2',
        'sales' => 'heroicon-o-chart-bar',
        'accountant' => 'heroicon-o-calculator',
        'warehouse_keeper' => 'heroicon-o-archive-box',
        'driver' => 'heroicon-o-truck',
        'planning' => 'heroicon-o-swatch',
    ];

    public function getLabel(): ?string
    {
        return self::LABELS[$this->value] ?? $this->value;
    }

    public function getColor(): string|array|null
    {
        return self::COLORS[$this->value] ?? 'gray';
    }

    public function getIcon(): ?string
    {
        return self::ICONS[$this->value] ?? null;
    }

    /** خيارات جاهزة لـ Select: ['admin' => 'المدير العام', ...] */
    public static function options(): array
    {
        $out = [];
        foreach (self::cases() as $case) {
            $out[$case->value] = $case->getLabel() ?? $case->value;
        }

        return $out;
    }

    /** آمن: يحول string إلى enum أو null */
    public static function tryFromValue(?string $value): ?self
    {
        return $value ? self::tryFrom($value) : null;
    }

    /**
     * مفيد مع Spatie:
     * - إذا عندك مستخدم متعدد الأدوار، مرّر اسم الدور الأول أو أي دور تختاره
     */
    public static function fromSpatieRoleName(?string $roleName): ?self
    {
        return self::tryFromValue($roleName);
    }

    /**
     * بيانات Badge موحّدة (تفيد بالعرض)
     * ترجع: ['label' => ..., 'color' => ..., 'icon' => ...]
     */
    public function badge(): array
    {
        return [
            'label' => $this->getLabel() ?? $this->value,
            'color' => $this->getColor() ?? 'gray',
            'icon' => $this->getIcon(),
        ];
    }

    /** جميع القيم */
    public static function values(): array
    {
        return array_map(static fn (self $c) => $c->value, self::cases());
    }
}
