<?php

namespace App\Enums;

enum ProductUnit: string
{
    case PIECE  = 'piece';      // قطعة
    case METER  = 'meter';      // متر
    case SET    = 'set';          // طقم
    case PAIR   = 'pair';        // زوج
    case BOX    = 'box';          // صندوق
    case PACK   = 'pack';        // علبة
    case KG     = 'kg';          // كيلوغرام
    case GRAM   = 'gram';        // غرام
    case LITER  = 'liter';      // لتر

    /**
     * الحصول على التسمية بالعربية
     */
    public function label(): string
    {
        return match ($this) {
            self::PIECE     => 'قطعة',
            self::METER     => 'متر',
            self::SET       => 'طقم',
            self::PAIR      => 'زوج',
            self::BOX       => 'صندوق',
            self::PACK      => 'علبة',
            self::KG        => 'كيلوغرام',
            self::GRAM      => 'غرام',
            self::LITER     => 'لتر',
        };
    }

    /**
     * الحصول على الرمز المختصر
     */
    public function symbol(): string
    {
        return match ($this) {
            self::PIECE     => 'قطعة',
            self::METER     => 'م',
            self::SET       => 'طقم',
            self::PAIR      => 'زوج',
            self::BOX       => 'صندوق',
            self::PACK      => 'علبة',
            self::KG        => 'كغ',
            self::GRAM      => 'غ',
            self::LITER     => 'ل',
        };
    }

    /**
     * الحصول على الوصف
     */
    public function description(): string
    {
        return match ($this) {
            self::PIECE     => 'للمنتجات التي تُباع بالقطعة الواحدة',
            self::METER     => 'للمنتجات التي تُباع بالمتر (أثاث، كاونترات)',
            self::SET       => 'للمنتجات التي تُباع كطقم كامل',
            self::PAIR      => 'للمنتجات التي تُباع بالزوج (أحذية، جوارب)',
            self::BOX       => 'للمنتجات المعبأة في صناديق',
            self::PACK      => 'للمنتجات المعبأة في علبة',
            self::KG        => 'للمنتجات التي تُباع بالوزن (كيلوغرام)',
            self::GRAM      => 'للمنتجات الصغيرة التي تُباع بالغرام',
            self::LITER     => 'للسوائل التي تُباع باللتر',
        };
    }

    /**
     * الحصول على الأيقونة
     */
    public function icon(): string
    {
        return match ($this) {
            self::PIECE     => 'heroicon-o-cube',
            self::METER     => 'heroicon-o-arrow-trending-up',
            self::SET       => 'heroicon-o-squares-2x2',
            self::PAIR      => 'heroicon-o-arrows-right-left',
            self::BOX       => 'heroicon-o-archive-box',
            self::PACK      => 'heroicon-o-cube-transparent',
            self::KG        => 'heroicon-o-scale',
            self::GRAM      => 'heroicon-o-beaker',
            self::LITER     => 'heroicon-o-beaker',
        };
    }

    /**
     * الحصول على اللون
     */
    public function color(): string
    {
        return match ($this) {
            self::PIECE     => 'primary',
            self::METER     => 'info',
            self::SET       => 'success',
            self::PAIR      => 'warning',
            self::BOX       => 'gray',
            self::PACK      => 'gray',
            self::KG        => 'danger',
            self::GRAM      => 'danger',
            self::LITER     => 'info',
        };
    }

    /**
     * الحصول على جميع الخيارات لـ Filament
     */
    public static function options(): array
    {
        return collect(self::cases())
            ->mapWithKeys(fn($case) => [$case->value => $case->label()])
            ->toArray();
    }

    /**
     * الحصول على جميع الخيارات مع الوصف
     */
    public static function optionsWithDescription(): array
    {
        return collect(self::cases())
            ->mapWithKeys(fn($case) => [
                $case->value => $case->label() . ' - ' . $case->description()
            ])
            ->toArray();
    }
}
