<?php

namespace App\Enums;

/**
 * ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
 * حالات الطلب للمتجر الإلكتروني
 * ━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
 */
enum OrderStatus: string
{
    case Pending = 'pending';       // قيد الانتظار
    case Confirmed = 'confirmed';     // تم التأكيد
    case Processing = 'processing';    // قيد المعالجة
    case ReadyToShip = 'ready_to_ship'; // جاهز للشحن
    case Shipped = 'shipped';       // تم الشحن
    case Delivered = 'delivered';     // تم التوصيل
    case Completed = 'completed';     // مكتمل
    case Canceled = 'canceled';      // ملغي
    case Refunded = 'refunded';      // تم الاسترجاع

    public function label(): string
    {
        return match ($this) {
            self::Pending => 'قيد الانتظار',
            self::Confirmed => 'تم التأكيد',
            self::Processing => 'قيد المعالجة',
            self::ReadyToShip => 'جاهز للشحن',
            self::Shipped => 'تم الشحن',
            self::Delivered => 'تم التوصيل',
            self::Completed => 'مكتمل',
            self::Canceled => 'ملغي',
            self::Refunded => 'تم الاسترجاع',
        };
    }

    public function color(): string
    {
        return match ($this) {
            self::Pending => 'gray',
            self::Confirmed => 'info',
            self::Processing => 'warning',
            self::ReadyToShip => 'primary',
            self::Shipped => 'info',
            self::Delivered => 'success',
            self::Completed => 'success',
            self::Canceled => 'danger',
            self::Refunded => 'gray',
        };
    }

    public function icon(): string
    {
        return match ($this) {
            self::Pending => 'heroicon-o-clock',
            self::Confirmed => 'heroicon-o-check-circle',
            self::Processing => 'heroicon-o-cog-6-tooth',
            self::ReadyToShip => 'heroicon-o-cube',
            self::Shipped => 'heroicon-o-truck',
            self::Delivered => 'heroicon-o-home',
            self::Completed => 'heroicon-o-check-badge',
            self::Canceled => 'heroicon-o-x-circle',
            self::Refunded => 'heroicon-o-arrow-uturn-left',
        };
    }

    public static function options(): array
    {
        return collect(self::cases())
            ->mapWithKeys(fn (self $status) => [$status->value => $status->label()])
            ->all();
    }

    /**
     * الحالات التي يمكن إلغاء الطلب فيها
     */
    public function isCancelable(): bool
    {
        return in_array($this, [
            self::Pending,
            self::Confirmed,
            self::Processing,
            self::ReadyToShip,
            self::Shipped,
        ]);
    }

    /**
     * الحالات النشطة (الطلب قيد التنفيذ)
     */
    public function isActive(): bool
    {
        return in_array($this, [
            self::Pending,
            self::Confirmed,
            self::Processing,
            self::ReadyToShip,
            self::Shipped,
        ]);
    }

    /**
     * الحالات المنتهية
     */
    public function isFinal(): bool
    {
        return in_array($this, [
            self::Delivered,
            self::Completed,
            self::Canceled,
            self::Refunded,
        ]);
    }
}
