<?php

namespace App\Console\Commands\Products;

use App\Enums\MovementDirection;
use App\Models\Attribute;
use App\Models\AttributeValue;
use App\Models\Brand;
use App\Models\Category;
use App\Models\Product;
use App\Models\ProductPrice;
use App\Models\ProductVariant;
use App\Models\PurchaseInvoice;
use App\Models\PurchaseInvoiceItem;
use App\Models\PurchaseInvoiceItemVariant;
use App\Models\PurchaseInvoiceItemVariantAttributeValue;
use App\Models\StockMovement;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use ZipArchive;

class ImportProductsImages extends Command
{
    protected $signature = 'purchases:import-products-images
        {zip : مسار ملف ZIP مثل storage/app/import/products.zip}
        {--branch_id=1}
        {--warehouse_id=1}
        {--supplier_id=1}
        {--user_id=1}
        {--invoice_number= : رقم الفاتورة (اختياري)}
        {--dry : تجربة بدون حفظ}
    ';

    protected $description = 'استيراد منتجات + صور (Thumbnail + Gallery) + متغيرات + تسعير + فاتورة شراء + مخزون';

    /** @var array<string, Attribute> */
    protected array $attrCache = [];

    /** @var array<string, AttributeValue> */
    protected array $attrValueCache = [];

    public function handle(): int
    {
        $zipPath = (string) $this->argument('zip');

        if (! file_exists($zipPath)) {
            $this->error("❌ الملف غير موجود: {$zipPath}");

            return self::FAILURE;
        }

        $dry = (bool) $this->option('dry');
        $branchId = (int) $this->option('branch_id');
        $warehouseId = (int) $this->option('warehouse_id');
        $supplierId = (int) $this->option('supplier_id');
        $userId = (int) $this->option('user_id');

        $tmpDir = storage_path('app/tmp/import-'.Str::uuid());
        File::ensureDirectoryExists($tmpDir);

        try {
            $this->info('📦 فك الضغط...');
            $this->extractZip($zipPath, $tmpDir);

            $files = collect(File::allFiles($tmpDir))
                ->filter(fn ($f) => in_array(strtolower($f->getExtension()), ['jpg', 'jpeg', 'png', 'webp'], true))
                ->values();

            if ($files->isEmpty()) {
                $this->warn('⚠️ لا توجد صور صالحة داخل ZIP.');

                return self::SUCCESS;
            }

            $groups = $files->groupBy(fn ($f) => $this->baseKeyFromFilename($f->getFilename()));
            $this->info('🧩 عدد المنتجات: '.$groups->count());

            if ($dry) {
                foreach ($groups as $base => $imgs) {
                    $this->line("DRY ▶ {$base} | imgs={$imgs->count()}");
                }

                return self::SUCCESS;
            }

            [$categoryId, $brandId, $attrColor, $attrSize] = $this->ensureCatalog();

            $invoiceNumber = (string) ($this->option('invoice_number')
                ?: 'PUR-'.now()->format('Ymd').'-'.str_pad((string) random_int(1, 99999), 5, '0', STR_PAD_LEFT));

            $invoice = PurchaseInvoice::create([
                'invoice_number' => $invoiceNumber,
                'supplier_id' => $supplierId,
                'branch_id' => $branchId,
                'warehouse_id' => $warehouseId,
                'user_id' => $userId,
                'invoice_date' => now()->toDateString(),
                'due_date' => now()->addDays(7)->toDateString(),
                'subtotal' => 0,
                'discount' => 0,
                'freight' => 0,
                'total' => 0,
                'paid' => 0,
                'due' => 0,
                'status' => 'received',
                'payment_status' => 'unpaid',
                'notes' => 'مستوردة تلقائياً من ZIP صور',
            ]);

            $subtotal = 0;
            $this->info('🚀 بدء الاستيراد...');
            $bar = $this->output->createProgressBar($groups->count());
            $bar->start();

            foreach ($groups as $baseCode => $images) {

                $thumb = $images->first(fn ($f) => ! $this->isAlt($f->getFilename())) ?: $images->first();
                $gallery = $images->filter(fn ($f) => $thumb && $f->getPathname() !== $thumb->getPathname())->values();

                $colors = $this->extractColorCodes($images);
                if (count($colors) < 1) {
                    $colors = ['BLK', 'WHT'];
                }
                $sizes = ['M', 'L'];

                $productName = $this->buildArabicProductName((string) $baseCode, $images);
                $baseUnitCost = (int) $this->randCost();

                DB::transaction(function () use (
                    &$subtotal,
                    $invoice,
                    $categoryId,
                    $brandId,
                    $attrColor,
                    $attrSize,
                    $productName,
                    $thumb,
                    $gallery,
                    $colors,
                    $sizes,
                    $baseUnitCost,
                    $warehouseId,
                    $userId
                ) {

                    $product = Product::query()->firstOrCreate(
                        ['name' => $productName],
                        [
                            'description' => null,
                            'category_id' => $categoryId,
                            'brand_id' => $brandId,
                            'unit' => 'piece',
                            'is_variable' => true,
                            'is_active' => true,
                        ]
                    );

                    if ($thumb) {
                        $product->addMedia($thumb->getPathname())
                            ->toMediaCollection(Product::MEDIA_COLLECTION_THUMBNAIL);
                    }

                    if ($gallery->isNotEmpty()) {
                        $this->syncGalleryMedia($product, $gallery);
                    }

                    $variantList = [];
                    foreach ($colors as $c) {
                        foreach ($sizes as $s) {
                            $variantList[] = [$c, $s];
                        }
                    }
                    if (count($variantList) < 2) {
                        $variantList = [[$colors[0], 'M'], [$colors[0], 'L']];
                    }

                    $firstVariantId = null;
                    $totalQty = 0;
                    $item = null;

                    foreach ($variantList as [$colorCode, $sizeCode]) {

                        $colorCode = strtoupper((string) $colorCode);
                        $sizeCode = strtoupper((string) $sizeCode);

                        $colorValue = $this->ensureColorValueCached($attrColor, $colorCode);
                        $sizeValue = $this->ensureSizeValueCached($attrSize, $sizeCode);

                        $variant = $this->findVariantByAttributes(
                            $product->id,
                            $attrColor->id,
                            $colorValue->id,
                            $attrSize->id,
                            $sizeValue->id
                        );

                        if (! $variant) {
                            $variant = ProductVariant::create([
                                'product_id' => $product->id,
                                'cost' => $baseUnitCost,
                                'is_active' => true,
                                'is_default' => $firstVariantId === null,
                            ]);

                            $variant->attributeValues()->syncWithoutDetaching([
                                $colorValue->id => ['attribute_id' => $attrColor->id],
                                $sizeValue->id => ['attribute_id' => $attrSize->id],
                            ]);
                        } else {
                            $variant->update([
                                'is_active' => true,
                                'cost' => max($variant->cost, $baseUnitCost),
                            ]);
                        }

                        if ($firstVariantId === null) {
                            $firstVariantId = $variant->id;
                            $item = PurchaseInvoiceItem::create([
                                'purchase_invoice_id' => $invoice->id,
                                'product_variant_id' => $firstVariantId,
                                'quantity_ordered' => 0,
                                'quantity_received' => 0,
                                'unit_cost' => $baseUnitCost,
                                'line_total' => 0,
                                'receive_status' => 'received',
                            ]);
                        }

                        ProductPrice::updateOrCreate(
                            ['product_id' => $product->id, 'product_variant_id' => $variant->id],
                            [
                                'retail_margin' => 20.0,
                                'wholesale_margin' => 10.0,
                                'agent_margin' => 5.0,
                                'retail_price' => 0,
                                'wholesale_price' => 0,
                                'agent_price' => 0,
                                'status' => ProductPrice::STATUS_ACTIVE,
                            ]
                        );

                        $qty = random_int(5, 25);
                        $totalQty += $qty;

                        $row = PurchaseInvoiceItemVariant::create([
                            'purchase_invoice_item_id' => $item->id,
                            'product_variant_id' => $variant->id,
                            'quantity' => $qty,
                        ]);

                        PurchaseInvoiceItemVariantAttributeValue::updateOrCreate(
                            ['purchase_invoice_item_variant_id' => $row->id, 'attribute_id' => $attrColor->id],
                            ['attribute_value_id' => $colorValue->id]
                        );
                        PurchaseInvoiceItemVariantAttributeValue::updateOrCreate(
                            ['purchase_invoice_item_variant_id' => $row->id, 'attribute_id' => $attrSize->id],
                            ['attribute_value_id' => $sizeValue->id]
                        );

                        StockMovement::create([
                            'warehouse_id' => $warehouseId,
                            'product_variant_id' => $variant->id,
                            'direction' => MovementDirection::IN,
                            'source_type' => PurchaseInvoice::class,
                            'source_id' => $invoice->id,
                            'quantity' => $qty,
                            'unit_cost' => $baseUnitCost,
                            'total_cost' => $qty * $baseUnitCost,
                            'moved_at' => now(),
                            'reference' => $invoice->invoice_number,
                            'user_id' => $userId,
                        ]);
                    }

                    $item->update([
                        'quantity_ordered' => $totalQty,
                        'quantity_received' => $totalQty,
                        'unit_cost' => $baseUnitCost,
                        'line_total' => $totalQty * $baseUnitCost,
                    ]);

                    $subtotal += $totalQty * $baseUnitCost;
                });

                $bar->advance();
            }

            $bar->finish();
            $this->newLine();

            $invoice->update([
                'subtotal' => $subtotal,
                'total' => $subtotal,
                'paid' => 0,
                'due' => $subtotal,
            ]);

            $this->info("✅ تم إنشاء الفاتورة بنجاح: {$invoice->invoice_number}");

            return self::SUCCESS;
        } finally {
            File::deleteDirectory($tmpDir);
        }
    }

    /* =========================
     * Helpers الكاملة
     * ========================= */

    private function extractZip(string $zipPath, string $toDir): void
    {
        $zip = new ZipArchive;
        if ($zip->open($zipPath) !== true) {
            throw new \RuntimeException('فشل فتح ZIP.');
        }
        $zip->extractTo($toDir);
        $zip->close();
    }

    private function isAlt(string $filename): bool
    {
        return str_contains(strtoupper($filename), '_ALT_');
    }

    private function baseKeyFromFilename(string $filename): string
    {
        $name = pathinfo($filename, PATHINFO_FILENAME);
        $name = preg_replace('/\s*\(\d+\)\s*$/', '', $name);
        if (stripos($name, '_ALT_') !== false) {
            $name = explode('_ALT_', $name)[0];
        }
        $parts = explode('_', $name);
        if (count($parts) >= 3) {
            $last = end($parts);
            $isUuidLike = str_contains($last, '-') && strlen($last) >= 16;
            $isLongHex = strlen($last) >= 16 && ctype_xdigit(str_replace('-', '', $last));
            if ($isUuidLike || $isLongHex) {
                array_pop($parts);
            }
        }

        return implode('_', $parts);
    }

    private function extractColorCodes($images): array
    {
        $codes = [];
        foreach ($images as $f) {
            $n = pathinfo($f->getFilename(), PATHINFO_FILENAME);
            if (stripos($n, '_ALT_') !== false) {
                $n = explode('_ALT_', $n)[0];
            }
            $p = explode('_', $n);
            $last = strtoupper(end($p));
            if (strlen($last) >= 3 && strlen($last) <= 5) {
                $codes[] = $last;
            }
        }

        return array_values(array_unique($codes));
    }

    private function randCost(): int
    {
        return random_int(2000, 8000);
    }

    private function syncGalleryMedia(Product $product, $galleryFiles): void
    {
        $product->loadMissing('media');
        $existingNames = $product->getMedia(Product::MEDIA_COLLECTION_GALLERY)
            ->pluck('file_name')->filter()->map(fn ($v) => strtolower((string) $v))->all();

        foreach ($galleryFiles as $img) {
            $fileName = strtolower((string) $img->getFilename());
            if (in_array($fileName, $existingNames, true)) {
                continue;
            }
            $product->addMedia($img->getPathname())
                ->usingFileName($img->getFilename())
                ->toMediaCollection(Product::MEDIA_COLLECTION_GALLERY);
            $existingNames[] = $fileName;
        }
    }

    private function findVariantByAttributes(int $productId, int $attrColorId, int $colorValueId, int $attrSizeId, int $sizeValueId): ?ProductVariant
    {
        $variantId = DB::table('product_variants as pv')
            ->join('variant_attribute_values as v1', function ($j) use ($attrColorId, $colorValueId) {
                $j->on('v1.variant_id', '=', 'pv.id')
                    ->where('v1.attribute_id', '=', $attrColorId)
                    ->where('v1.attribute_value_id', '=', $colorValueId);
            })
            ->join('variant_attribute_values as v2', function ($j) use ($attrSizeId, $sizeValueId) {
                $j->on('v2.variant_id', '=', 'pv.id')
                    ->where('v2.attribute_id', '=', $attrSizeId)
                    ->where('v2.attribute_value_id', '=', $sizeValueId);
            })
            ->where('pv.product_id', $productId)
            ->select('pv.id')
            ->value('pv.id');

        return $variantId ? ProductVariant::find((int) $variantId) : null;
    }

    private function ensureCatalog(): array
    {
        $cat = Category::firstOrCreate(['name' => 'الجوارب'], ['is_active' => true]);
        $brand = Brand::firstOrCreate(['name' => 'Nassaj'], ['is_active' => true]);

        $attrColor = $this->ensureAttributeCached('color', 'اللون', 'color');
        $attrSize = $this->ensureAttributeCached('size', 'المقاس', 'text');

        foreach (
            [
                'BLK' => ['أسود', '#111827'],
                'WHT' => ['أبيض', '#F9FAFB'],
                'PNK' => ['وردي', '#EC4899'],
                'BLU' => ['أزرق', '#3B82F6'],
                'GRY' => ['رمادي', '#6B7280'],
            ] as $code => [$label, $hex]
        ) {
            $this->ensureAttributeValueCached($attrColor, $code, $label, $hex);
        }

        foreach (['S', 'M', 'L', 'XL'] as $s) {
            $this->ensureAttributeValueCached($attrSize, $s, strtoupper($s), null);
        }

        return [$cat->id, $brand->id, $attrColor, $attrSize];
    }

    private function buildArabicProductName(string $baseCode, $images): string
    {
        $s = Str::lower($baseCode);
        $celebrity = $this->arabicCelebrityName($s);
        if ($celebrity !== '') {
            return trim("شراب {$celebrity}");
        }

        $type = match (true) {
            str_contains($s, 'perf') || str_contains($s, 'performance') => 'جوارب رياضية',
            str_contains($s, 'casual') => 'جوارب كاجوال',
            str_contains($s, 'winter') || str_contains($s, 'wool') || str_contains($s, 'snow') => 'جوارب شتوية',
            str_contains($s, 'light') || str_contains($s, 'thin') => 'جوارب خفيفة',
            str_contains($s, 'women') || str_contains($s, 'lady') || str_contains($s, 'dona') => 'جوارب نسائية',
            str_contains($s, 'kids') || str_contains($s, 'child') => 'جوارب أطفال',
            default => 'جوارب',
        };

        $style = match (true) {
            str_contains($s, 'stripe') || str_contains($s, 'stripes') => 'مخططة',
            str_contains($s, 'plain') || str_contains($s, 'solid') => 'سادة',
            str_contains($s, 'dots') || str_contains($s, 'polka') => 'منقطة',
            str_contains($s, 'logo') => 'بشعار',
            default => '',
        };

        $colors = $this->extractColorCodes($images);
        $colorName = $this->arabicColorName($colors[0] ?? null);

        $name = trim(implode(' ', array_filter([$type, $style])));
        if ($colorName !== '') {
            $name .= ' '.$colorName;
        }
        $name = preg_replace('/[A-Za-z0-9_]+/u', '', $name);
        $name = preg_replace('/\s{2,}/u', ' ', $name);

        return trim((string) $name);
    }

    private function arabicColorName(?string $code): string
    {
        $code = strtoupper((string) $code);

        return match ($code) {
            'GRY', 'GREY', 'GRAY' => 'رمادي',
            'BLK', 'BLACK' => 'أسود',
            'WHT', 'WHITE' => 'أبيض',
            'PNK', 'PINK' => 'وردي',
            'BLU', 'BLUE' => 'أزرق',
            'RED' => 'أحمر',
            'GRN' => 'أخضر',
            'BRN' => 'بني',
            default => '',
        };
    }

    private function arabicCelebrityName(string $s): string
    {
        return match (true) {
            str_contains($s, 'drake') => 'دريك',
            str_contains($s, 'travis') || str_contains($s, 'travisscott') || str_contains($s, 'scott') => 'ترافيس سكوت',
            default => '',
        };
    }

    private function ensureAttributeCached(string $name, string $display, string $type): Attribute
    {
        if (isset($this->attrCache[$name])) {
            return $this->attrCache[$name];
        }
        $attr = Attribute::firstOrCreate(['name' => $name], ['label' => $display, 'type' => $type, 'is_active' => true]);
        $this->attrCache[$name] = $attr;

        return $attr;
    }

    private function ensureAttributeValueCached(Attribute $attr, string $code, string $label, ?string $hex = null): AttributeValue
    {
        $key = $attr->name.'-'.$code;
        if (isset($this->attrValueCache[$key])) {
            return $this->attrValueCache[$key];
        }
        $val = AttributeValue::firstOrCreate(
            ['attribute_id' => $attr->id, 'value' => $code],
            ['display_value' => $label, 'color_code' => $hex]
        );
        $this->attrValueCache[$key] = $val;

        return $val;
    }

    private function ensureColorValueCached(Attribute $attr, string $code): AttributeValue
    {
        return $this->ensureAttributeValueCached($attr, $code, $this->arabicColorName($code), null);
    }

    private function ensureSizeValueCached(Attribute $attr, string $code): AttributeValue
    {
        return $this->ensureAttributeValueCached($attr, $code, strtoupper($code), null);
    }
}
