<?php

namespace App\Console\Commands;

use App\Models\Employee;
use App\Models\Payroll;
use App\Models\PayrollRun;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class GeneratePayrollRun extends Command
{
    /**
     * الاستخدام:
     * php artisan payroll:generate 1 2026 1
     * php artisan payroll:generate 1 2026 1 --include-daily
     */
    protected $signature = 'payroll:generate
        {branch_id : رقم الفرع}
        {year : السنة مثال 2026}
        {month : الشهر 1-12}
        {--include-daily : تضمين الموظفين اليوميين (basic_salary = 0)}';

    protected $description = 'إنشاء دورة رواتب لشهر محدد وإنشاء كشوف رواتب للموظفين النشطين في الفرع';

    public function handle(): int
    {
        $branchId = (int) $this->argument('branch_id');
        $year     = (int) $this->argument('year');
        $month    = (int) $this->argument('month');

        if ($branchId <= 0) {
            $this->error('branch_id غير صحيح.');
            return self::FAILURE;
        }

        if ($year < 2000 || $year > 2100) {
            $this->error('year غير صحيح.');
            return self::FAILURE;
        }

        if ($month < 1 || $month > 12) {
            $this->error('الشهر يجب أن يكون بين 1 و 12.');
            return self::FAILURE;
        }

        $includeDaily = (bool) $this->option('include-daily');

        $createdPayrolls = 0;
        $existingPayrolls = 0;
        $employeesCount = 0;

        try {
            DB::transaction(function () use ($branchId, $year, $month, $includeDaily, &$createdPayrolls, &$existingPayrolls, &$employeesCount) {

                // 1) إنشاء/جلب دورة الرواتب
                $run = PayrollRun::query()->firstOrCreate(
                    [
                        'branch_id' => $branchId,
                        'year'      => $year,
                        'month'     => $month,
                    ],
                    [
                        'is_locked' => false,
                        'locked_at' => null,
                    ]
                );

                if ((bool) $run->is_locked) {
                    throw new \RuntimeException('هذه الدورة مقفلة ولا يمكن توليد كشوف جديدة.');
                }

                // 2) جلب الموظفين النشطين في الفرع
                $employeesQ = Employee::query()
                    ->where('branch_id', $branchId)
                    ->where('is_active', true);

                if (! $includeDaily) {
                    $employeesQ->where('salary_type', Employee::SALARY_MONTHLY);
                }

                $employees = $employeesQ->with('payGrade')->get();
                $employeesCount = $employees->count();

                // 3) إنشاء كشف راتب لكل موظف
                foreach ($employees as $employee) {

                    // Snapshot للراتب الأساسي لذلك الشهر (INT)
                    $basicSalary = 0;

                    if ((string) $employee->salary_type === Employee::SALARY_MONTHLY) {
                        $basicSalary = (int) ($employee->payGrade?->amount ?? 0);
                    }

                    $payroll = Payroll::query()->firstOrCreate(
                        [
                            'employee_id' => (int) $employee->id,
                            'year'        => $year,
                            'month'       => $month,
                        ],
                        [
                            'branch_id'       => $branchId,
                            'payroll_run_id'  => (int) $run->id,

                            'basic_salary'    => $basicSalary,
                            'total_bonus'     => 0,
                            'total_deduction' => 0,
                            'net_salary'      => $basicSalary,

                            'is_paid'         => false,
                            'paid_at'         => null,
                        ]
                    );

                    if ($payroll->wasRecentlyCreated) {
                        $createdPayrolls++;
                        continue;
                    }

                    $existingPayrolls++;

                    // في حال كان موجوداً سابقاً: نضمن الربط الصحيح
                    $changed = false;

                    if ((int) $payroll->branch_id !== $branchId) {
                        $payroll->branch_id = $branchId;
                        $changed = true;
                    }

                    if ((int) $payroll->payroll_run_id !== (int) $run->id) {
                        $payroll->payroll_run_id = (int) $run->id;
                        $changed = true;
                    }

                    // لا نغيّر basic_salary إذا كان موجوداً مسبقاً (Snapshot)
                    // لكن إذا كان 0 والموظف شهري ويمتلك payGrade، نصلحه
                    if ((int) $payroll->basic_salary === 0 && (string) $employee->salary_type === Employee::SALARY_MONTHLY && $basicSalary > 0) {
                        $payroll->basic_salary = $basicSalary;
                        $changed = true;
                    }

                    if ($changed) {
                        $payroll->save();
                        $payroll->recalculate();
                    }
                }
            });
        } catch (\Throwable $e) {
            $this->error('فشلت العملية: ' . $e->getMessage());
            return self::FAILURE;
        }

        $this->info('تمت العملية بنجاح.');
        $this->line('الفرع: ' . $branchId . ' | الفترة: ' . sprintf('%04d/%02d', $year, $month));
        $this->line('عدد الموظفين المشمولين: ' . number_format((int) $employeesCount));
        $this->line('كشوف جديدة: ' . number_format((int) $createdPayrolls));
        $this->line('كشوف موجودة مسبقاً: ' . number_format((int) $existingPayrolls));

        return self::SUCCESS;
    }
}
