<?php

namespace Elementor;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class OSF_Elementor_Price_Table extends Widget_Base {

    public function get_name() {
        return 'opal-price-table';
    }

    public function get_title() {
        return __('Opal Price Table', 'konstruktic-core');
    }

    public function get_categories() {
        return array('opal-addons');
    }

    public function get_icon() {
        return 'eicon-price-table';
    }

    protected function register_controls() {
        $this->start_controls_section(
            'section_header',
            [
                'label' => __('Header', 'konstruktic-core'),
            ]
        );

        $this->add_control(
            'heading',
            [
                'label' => __('Title', 'konstruktic-core'),
                'type' => Controls_Manager::TEXT,
                'default' => __('Pricing Table', 'konstruktic-core'),
            ]
        );

        $this->add_control(
            'icon',
            [
                'label' => __( 'Choose Icon', 'konstruktic-core' ),
                'type' => Controls_Manager::ICON,
                'default' => '',
            ]
        );

        $this->add_control(
            'image',
            [
                'label'     => __('Background Image', 'konstruktic-core'),
                'type'      => Controls_Manager::MEDIA,
                'dynamic'   => [
                    'active' => true,
                ],
            ]
        );

        $this->add_control(
            'heading_description',
            [
                'label' => __('Description', 'konstruktic-core'),
                'type' => Controls_Manager::TEXTAREA,
                'placeholder' => __('Enter your description', 'konstruktic-core'),
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_pricing',
            [
                'label' => __('Pricing', 'konstruktic-core'),
            ]
        );

        $this->add_control(
            'price',
            [
                'label' => __('Price', 'konstruktic-core'),
                'type' => Controls_Manager::NUMBER,
                'default' => '39.99',
            ]
        );

        $this->add_control(
            'period',
            [
                'label' => __('Period', 'konstruktic-core'),
                'type' => Controls_Manager::TEXT,
                'default' => __('month', 'konstruktic-core'),
                'placeholder'   => __('Period ...','konstruktic-core'),
            ]
        );

        $this->add_control(
            'currency_symbol',
            [
                'label' => __('Currency Symbol', 'konstruktic-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '' => __('None', 'konstruktic-core'),
                    'dollar' => '&#36; ' . _x('Dollar', 'Currency Symbol', 'konstruktic-core'),
                    'euro' => '&#128; ' . _x('Euro', 'Currency Symbol', 'konstruktic-core'),
                    'baht' => '&#3647; ' . _x('Baht', 'Currency Symbol', 'konstruktic-core'),
                    'franc' => '&#8355; ' . _x('Franc', 'Currency Symbol', 'konstruktic-core'),
                    'guilder' => '&fnof; ' . _x('Guilder', 'Currency Symbol', 'konstruktic-core'),
                    'krona' => 'kr ' . _x('Krona', 'Currency Symbol', 'konstruktic-core'),
                    'lira' => '&#8356; ' . _x('Lira', 'Currency Symbol', 'konstruktic-core'),
                    'peseta' => '&#8359 ' . _x('Peseta', 'Currency Symbol', 'konstruktic-core'),
                    'peso' => '&#8369; ' . _x('Peso', 'Currency Symbol', 'konstruktic-core'),
                    'pound' => '&#163; ' . _x('Pound Sterling', 'Currency Symbol', 'konstruktic-core'),
                    'real' => 'R$ ' . _x('Real', 'Currency Symbol', 'konstruktic-core'),
                    'ruble' => '&#8381; ' . _x('Ruble', 'Currency Symbol', 'konstruktic-core'),
                    'rupee' => '&#8360; ' . _x('Rupee', 'Currency Symbol', 'konstruktic-core'),
                    'indian_rupee' => '&#8377; ' . _x('Rupee (Indian)', 'Currency Symbol', 'konstruktic-core'),
                    'shekel' => '&#8362; ' . _x('Shekel', 'Currency Symbol', 'konstruktic-core'),
                    'yen' => '&#165; ' . _x('Yen/Yuan', 'Currency Symbol', 'konstruktic-core'),
                    'won' => '&#8361; ' . _x('Won', 'Currency Symbol', 'konstruktic-core'),
                    'custom' => __('Custom', 'konstruktic-core'),
                ],
                'default' => 'dollar',
            ]
        );

        $this->add_control(
            'currency_symbol_custom',
            [
                'label' => __('Custom Symbol', 'konstruktic-core'),
                'type' => Controls_Manager::TEXT,
                'condition' => [
                    'currency_symbol' => 'custom',
                ],
            ]
        );

        $this->add_control(
            'currency_format',
            [
                'label' => __('Currency Format', 'konstruktic-core'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '' => '1,234.56 (Default)',
                    ',' => '1.234,56',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_features',
            [
                'label' => __('Features', 'konstruktic-core'),
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'item_text',
            [
                'label' => __('Text', 'konstruktic-core'),
                'type' => Controls_Manager::TEXT,
                'default' => __('List Item', 'konstruktic-core'),
            ]
        );

        $repeater->add_control(
            'item_check',
            [
                'label' => __('Check', 'konstruktic-core'),
                'type' => Controls_Manager::SWITCHER,
                'default'   => 'yes',
                'label_on' => 'Show',
                'label_off' => 'Hide',
            ]
        );

        $this->add_control(
            'features_list',
            [
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'item_text' => __('List Item #1', 'konstruktic-core'),
                    ],
                    [
                        'item_text' => __('List Item #2', 'konstruktic-core'),
                    ],
                    [
                        'item_text' => __('List Item #3', 'konstruktic-core'),
                    ],
                ],
                'title_field' => '{{{ item_text }}}',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_button',
            [
                'label' => __('Button', 'konstruktic-core'),
            ]
        );

        $this->add_control(
            'button_type',
            [
                'label' => __('Type', 'konstruktic-core'),
                'type' => Controls_Manager::SELECT,
                'default' => 'primary',
                'options' => [
                    'default' => __('Default', 'konstruktic-core'),
                    'primary' => __('Primary', 'konstruktic-core'),
                    'secondary' => __('Secondary', 'konstruktic-core'),
                    'outline_primary' => __('Outline Primary', 'konstruktic-core'),
                    'outline_secondary' => __('Outline Secondary', 'konstruktic-core'),
                    'link' => __('Link', 'konstruktic-core'),
                    'info' => __('Info', 'konstruktic-core'),
                    'success' => __('Success', 'konstruktic-core'),
                    'warning' => __('Warning', 'konstruktic-core'),
                    'danger' => __('Danger', 'konstruktic-core'),
                ],
                'prefix_class' => 'elementor-button-',
            ]
        );

        $this->add_control(
            'button_size',
            [
                'label' => __('Size', 'konstruktic-core'),
                'type' => Controls_Manager::SELECT,
                'default' => 'md',
                'options' => [
                    'xs' => __('Extra Small', 'konstruktic-core'),
                    'sm' => __('Small', 'konstruktic-core'),
                    'md' => __('Medium', 'konstruktic-core'),
                    'lg' => __('Large', 'konstruktic-core'),
                    'xl' => __('Extra Large', 'konstruktic-core'),
                ],
            ]
        );

        $this->add_control(
            'button_text',
            [
                'label' => __('Button Text', 'konstruktic-core'),
                'type' => Controls_Manager::TEXT,
                'default' => __('Click Here', 'konstruktic-core'),
            ]
        );

        $this->add_control(
            'link',
            [
                'label' => __('Link', 'konstruktic-core'),
                'type' => Controls_Manager::URL,
                'placeholder' => __('https://your-link.com', 'konstruktic-core'),
                'default' => [
                    'url' => '#',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_ribbon',
            [
                'label' => __( 'Ribbon', 'konstruktic-core' ),
            ]
        );

        $this->add_control(
            'show_ribbon',
            [
                'label'     => __( 'Show', 'konstruktic-core' ),
                'type'      => Controls_Manager::SWITCHER,
                'default'   => 'yes',
                'separator' => 'before',
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_wrapper_style',
            [
                'label' => __('Wrapper', 'konstruktic-core'),
                'tab' => Controls_Manager::TAB_STYLE,
                'show_label' => false,
            ]
        );

        $this->add_control(
            'wrapper_bkg',
            [
                'label' => __('Background', 'konstruktic-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table' => 'background: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'border_wrapper',
                'placeholder' => '1px',
                'default' => '1px',
                'selector' => '{{WRAPPER}} .elementor-price-table',
            ]
        );

        $this->add_control(
            'wrapper_border_radius',
            [
                'label' => __('Border Radius', 'konstruktic-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'wrapper_padding',
            [
                'label' => __('Padding', 'konstruktic-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'animation_moveup',
            [
                'label' => __('Effect Float', 'konstruktic-core'),
                'type'  => Controls_Manager::SWITCHER,
                'label_on' => 'Show',
                'label_off' => 'Hide',
                'separator' => 'before',
                'return_value'  => 'on',
                'prefix_class'  => 'elementor-price-table_eff-float-',
            ]
        );

        $this->end_controls_section();


        $this->start_controls_section(
            'price_table_heading_style',
            [
                'label' => __('Header', 'konstruktic-core'),
                'tab' => Controls_Manager::TAB_STYLE,
                'show_label' => false,
            ]
        );

        $this->add_control(
            'price_header_bkg',
            [
                'label' => __('Background', 'konstruktic-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__wrapper-header' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'image_opacity',
            [
                'label'     => __('Image Opacity', 'konstruktic-core'),
                'type'      => Controls_Manager::SLIDER,
                'range'     => [
                    'px' => [
                        'min' => 0,
                        'max' => 1,
                        'step' => 0.1,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__wrapper-header img' => 'opacity: {{SIZE}};',
                ],
            ]
        );

        $this->add_control(
            'image_blend_mode',
            [
                'label' => __('Image Blend Mode', 'konstruktic-core'),
                'type' => Controls_Manager::SELECT,
                'default' => '',
                'options' => [
                    '' => __('normal', 'konstruktic-core'),
                    'multiply' => __('Multiply', 'konstruktic-core'),
                    'screen' => __('Screen', 'konstruktic-core'),
                    'overlay' => __('Overlay', 'konstruktic-core'),
                    'darken' => __('Darken', 'konstruktic-core'),
                    'lighten' => __('Lighten', 'konstruktic-core'),
                    'color-dodge' => __('Color dodge', 'konstruktic-core'),
                    'color-burn' => __('Color burn', 'konstruktic-core'),
                    'difference' => __('Difference', 'konstruktic-core'),
                    'exclusion' => __('Exclusion', 'konstruktic-core'),
                    'hue' => __('Hue', 'konstruktic-core'),
                    'saturation' => __('Saturation', 'konstruktic-core'),
                    'color' => __('Color', 'konstruktic-core'),
                    'luminosity' => __('Luminosity', 'konstruktic-core'),
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__wrapper-header img'    => 'mix-blend-mode:{{VALUE}};'
                ]
            ]
        );



        $this->add_control(
            'price_header_alignment',
            [
                'label' => __('Alignment', 'konstruktic-core'),
                'type' => Controls_Manager::CHOOSE,
                'label_block' => false,
                'options' => [
                    'left' => [
                        'title' => __('Left', 'konstruktic-core'),
                        'icon' => 'fa fa-align-left',
                    ],
                    'center' => [
                        'title' => __('Center', 'konstruktic-core'),
                        'icon' => 'fa fa-align-center',
                    ],
                    'right' => [
                        'title' => __('Right', 'konstruktic-core'),
                        'icon' => 'fa fa-align-right',
                    ],
                ],
                'default'   => 'center',
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__wrapper-header' => 'text-align: {{VALUE}}',
                ],
                'prefix_class'  => 'elementor-price-table-',
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'price_header_border',
                'placeholder' => '1px',
                'default' => '1px',
                'selector' => '{{WRAPPER}} .elementor-price-table__wrapper-header',
            ]
        );

        $this->add_control(
            'price_header_border_radius',
            [
                'label' => __('Border Radius', 'konstruktic-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__wrapper-header' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'price_header_padding',
            [
                'label' => __('Padding', 'konstruktic-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__wrapper-header' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'icon_style',
            [
                'label' => __('Icon', 'konstruktic-core'),
                'type' => Controls_Manager::HEADING,
            ]
        );

        $this->add_control(
            'icon_color',
            [
                'label' => __('Color', 'konstruktic-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__icon-wrapper i' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'icon_bg_primary',
            [
                'label' => __('Background primary', 'konstruktic-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .price-table-icon-bkg' => 'background: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'icon_bg_secondary',
            [
                'label' => __('Background secondary', 'konstruktic-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__icon-wrapper .elementor-icon ' => 'background: {{VALUE}}; border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_size',
            [
                'label' => __( 'Size', 'konstruktic-core' ),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 200,
                    ],
                ],
                'default'   => [
                    'size'  => 125,
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__icon-wrapper .elementor-icon' => 'font-size: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_frame_size',
            [
                'label' => __( 'Framed Size', 'konstruktic-core' ),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 200,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__icon-wrapper .elementor-icon' => 'border-width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'heading_heading_style',
            [
                'label' => __('Title', 'konstruktic-core'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'heading_color',
            [
                'label' => __('Color', 'konstruktic-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__heading' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'heading_typography',
                'selector' => '{{WRAPPER}} .elementor-price-table__heading',
                'global' => [
                    'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_SECONDARY,
                ],
            ]
        );

        $this->add_control(
            'heading_spacing',
            [
                'label' => __('Spacing', 'konstruktic-core'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__heading' => 'margin-bottom: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_control(
            'heading_sub_heading_style',
            [
                'label' => __('Description', 'konstruktic-core'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'heading_description!'   => '',
                ]
            ]
        );

        $this->add_control(
            'sub_heading_color',
            [
                'label' => __('Color', 'konstruktic-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__heading-description' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'sub_heading_typography',
                'selector' => '{{WRAPPER}} .elementor-price-table__heading-description',
                'global' => [
                    'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_TEXT,
                ],
            ]
        );

        $this->add_control(
            'sub_heading_spacing',
            [
                'label' => __('Spacing', 'konstruktic-core'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__heading-description' => 'margin-bottom: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_pricing_element_style',
            [
                'label' => __('Pricing', 'konstruktic-core'),
                'tab' => Controls_Manager::TAB_STYLE,
                'show_label' => false,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'price_typo',
                'selector' => '{{WRAPPER}} .elementor-price-table__price',
            ]
        );

        $this->add_control(
            'sprice_spacing',
            [
                'label' => __('Spacing', 'konstruktic-core'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__price' => 'margin-bottom: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_control(
            'price_heaing_value',
            [
                'label' => __('Value', 'konstruktic-core'),
                'type' => Controls_Manager::HEADING,
            ]
        );

        $this->add_control(
            'price_color',
            [
                'label' => __('Color', 'konstruktic-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__price span' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'price_size',
            [
                'label' => __('Value Size', 'konstruktic-core'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__price' => 'font-size: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_control(
            'price_heaing_symbol',
            [
                'label' => __('Symbol', 'konstruktic-core'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'price_symbol_size',
            [
                'label' => __('Size', 'konstruktic-core'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                    ],
                ],
                'default'  => [
                    'size'  => 20,
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__currency' => 'font-size: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_control(
            'price_symbol_position',
            [
                'label' => __('Position', 'konstruktic-core'),
                'type' => Controls_Manager::CHOOSE,
                'label_block' => false,
                'default'   => 'flex-start',
                'options' => [
                    'flex-start' => [
                        'title' => __('Top', 'konstruktic-core'),
                        'icon' => 'eicon-v-align-top',
                    ],
                    'center' => [
                        'title' => __('Middle', 'konstruktic-core'),
                        'icon' => 'eicon-v-align-middle',
                    ],
                    'flex-end' => [
                        'title' => __('Bottom', 'konstruktic-core'),
                        'icon' => 'eicon-v-align-bottom',
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__currency' => 'align-self: {{VALUE}}',
                ],

            ]
        );

        $this->add_control(
            'price_heaing_period',
            [
                'label' => __('Period', 'konstruktic-core'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );


        $this->add_control(
            'price_period_color',
            [
                'label' => __('Color', 'konstruktic-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__period' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'price_period_typo',
                'selector' => '{{WRAPPER}} .elementor-price-table__period',
                'global' => [
                    'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_features_list_style',
            [
                'label' => __('Features', 'konstruktic-core'),
                'tab' => Controls_Manager::TAB_STYLE,
                'show_label' => false,
            ]
        );

        $this->add_responsive_control(
            'features_items_padding',
            [
                'label' => __('Items Padding', 'konstruktic-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__features-list .elementor-price-table__feature-inner' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'features_list_color_check',
            [
                'label' => __('Color Check', 'konstruktic-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__features-list .elementor-price-table__feature-inner.item_check' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'features_list_color_uncheck',
            [
                'label' => __('Color Uncheck', 'konstruktic-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__features-list .elementor-price-table__feature-inner:not(.item_check)' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'features_list_color_item',
            [
                'label' => __('Border Item Color', 'konstruktic-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__feature-inner' => 'border-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'features_list_typography',
                'selector' => '{{WRAPPER}} .elementor-price-table__features-list .elementor-price-table__feature-inner',
                'global' => [
                    'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_TEXT,
                ],
            ]
        );

        $this->add_control(
            'features_list_alignment',
            [
                'label' => __('Alignment', 'konstruktic-core'),
                'type' => Controls_Manager::CHOOSE,
                'label_block' => false,
                'options' => [
                    'flex-start' => [
                        'title' => __('Left', 'konstruktic-core'),
                        'icon' => 'fa fa-align-left',
                    ],
                    'center' => [
                        'title' => __('Center', 'konstruktic-core'),
                        'icon' => 'fa fa-align-center',
                    ],
                    'flex-end' => [
                        'title' => __('Right', 'konstruktic-core'),
                        'icon' => 'fa fa-align-right',
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__feature-inner' => 'justify-content: {{VALUE}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'features_list_padding',
            [
                'label' => __( 'Padding', 'konstruktic-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__features-list' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'icon_items',
            [
                'label' => __('Icon', 'konstruktic-core'),
                'type' => Controls_Manager::HEADING,
            ]
        );

        $this->add_responsive_control(
            'icon_items_size',
            [
                'label' => __( 'Size', 'konstruktic-core' ),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 20,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__feature-inner i' => 'font-size: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_items_spacing',
            [
                'label' => __( 'Spacing', 'konstruktic-core' ),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__feature-inner i' => 'margin-right: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_button_style',
            [
                'label' => __('Button', 'konstruktic-core'),
                'tab' => Controls_Manager::TAB_STYLE,
                'show_label' => false,
                'condition' => [
                    'button_text!' => '',
                ],
            ]
        );

        $this->add_control(
            'icon_indent',
            [
                'label'     => __('Icon Spacing', 'konstruktic-core'),
                'type'      => Controls_Manager::SLIDER,
                'range'     => [
                    'px' => [
                        'max' => 50,
                    ],
                ],
                'default'   => [
                    'size' => 7,
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table_button-icon' => 'margin-left: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'button_alignment',
            [
                'label' => __('Alignment', 'konstruktic-core'),
                'type' => Controls_Manager::CHOOSE,
                'label_block' => false,
                'default'   => 'justify',
                'options' => [
                    'left' => [
                        'title' => __('Left', 'konstruktic-core'),
                        'icon' => 'fa fa-align-left',
                    ],
                    'center' => [
                        'title' => __('Center', 'konstruktic-core'),
                        'icon' => 'fa fa-align-center',
                    ],
                    'right' => [
                        'title' => __('Right', 'konstruktic-core'),
                        'icon' => 'fa fa-align-right',
                    ],
                    'justify' => [
                        'title' => __('Justify', 'konstruktic-core'),
                        'icon' => 'fa fa-align-justify',
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__footer' => 'text-align: {{VALUE}}',
                ],
                'prefix_class'  => 'elementor-button-',
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'button_typography',
                'selector' => '{{WRAPPER}} .elementor-price-table__button',
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'border',
                'placeholder' => '1px',
                'default' => '1px',
                'selector' => '{{WRAPPER}} .elementor-button',
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'button_box_shadow',
                'selector' => '.elementor-price-table__button.elementor-button'
            ]
        );

        $this->add_control(
            'border_radius',
            [
                'label' => __('Border Radius', 'konstruktic-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} a.elementor-button, {{WRAPPER}} .elementor-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'text_padding',
            [
                'label' => __('Padding', 'konstruktic-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} a.elementor-button, {{WRAPPER}} .elementor-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->start_controls_tabs('tabs_button_style');

        $this->start_controls_tab(
            'tab_button_normal',
            [
                'label' => __('Normal', 'konstruktic-core'),
            ]
        );

        $this->add_control(
            'button_text_color',
            [
                'label' => __('Text Color', 'konstruktic-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__button' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_background_color',
            [
                'label' => __('Background Color', 'konstruktic-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__button' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_button_hover',
            [
                'label' => __('Hover', 'konstruktic-core'),
            ]
        );

        $this->add_control(
            'button_hover_color',
            [
                'label' => __('Text Color', 'konstruktic-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__button:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_background_hover_color',
            [
                'label' => __('Background Color', 'konstruktic-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__button:hover' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_hover_animation',
            [
                'label' => __('Animation', 'konstruktic-core'),
                'type' => Controls_Manager::HOVER_ANIMATION,
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();



        $this->end_controls_section();

        $this->start_controls_section(
            'section_ribbon_style',
            [
                'label'      => __( 'Ribbon', 'konstruktic-core' ),
                'tab'        => Controls_Manager::TAB_STYLE,
                'show_label' => false,
                'condition'  => [
                    'show_ribbon' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'ribbon_bg_color',
            [
                'label'     => __( 'Background', 'konstruktic-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__ribbon' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'ribbon_icon_color',
            [
                'label'     => __( 'Icon Color', 'konstruktic-core' ),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table-icon' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_footer_style',
            [
                'label' => __('Footer ', 'konstruktic-core'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'footer_background',
            [
                'label' => __('Background', 'konstruktic-core'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__footer' => 'background: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'footer_padding',
            [
                'label' => __('Padding', 'konstruktic-core'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table__footer' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_hover_color_style',
            [
                'label' => __('Color Hover', 'konstruktic-core'),
                'tab' => Controls_Manager::TAB_STYLE,
                'show_label' => false,
            ]
        );

        $this->add_control(
            'section_wrapper_bkg_hover',
            [
                'label' => __('Background Wrapper', 'konstruktic-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table:hover' => 'background-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'heading_color_hover',
            [
                'label' => __('Title', 'konstruktic-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table:hover .elementor-price-table__heading' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'border_color_hover',
            [
                'label' => __('Border Color', 'konstruktic-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table:hover .elementor-price-table__features-list' => 'border-color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'features_list_color_check_hover',
            [
                'label' => __('Color Check', 'konstruktic-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table:hover .elementor-price-table__features-list .elementor-price-table__feature-inner.item_check' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'features_list_color_uncheck_hover',
            [
                'label' => __('Color Uncheck', 'konstruktic-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table:hover .elementor-price-table__features-list .elementor-price-table__feature-inner:not(.item_check)' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'features_list_color_item_hover',
            [
                'label' => __('Border Item Color', 'konstruktic-core'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .elementor-price-table:hover .elementor-price-table__feature-inner' => 'border-color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();

    }

    private function get_currency_symbol($symbol_name) {
        $symbols = [
            'dollar' => '&#36;',
            'euro' => '&#128;',
            'franc' => '&#8355;',
            'pound' => '&#163;',
            'ruble' => '&#8381;',
            'shekel' => '&#8362;',
            'baht' => '&#3647;',
            'yen' => '&#165;',
            'won' => '&#8361;',
            'guilder' => '&fnof;',
            'peso' => '&#8369;',
            'peseta' => '&#8359',
            'lira' => '&#8356;',
            'rupee' => '&#8360;',
            'indian_rupee' => '&#8377;',
            'real' => 'R$',
            'krona' => 'kr',
        ];
        return isset($symbols[$symbol_name]) ? $symbols[$symbol_name] : '';
    }

    protected function render() {
        $settings = $this->get_settings();
        $symbol = '';

        if (!empty($settings['currency_symbol'])) {
            if ('custom' !== $settings['currency_symbol']) {
                $symbol = $this->get_currency_symbol($settings['currency_symbol']);
            } else {
                $symbol = $settings['currency_symbol_custom'];
            }
        }
        $currency_format = empty($settings['currency_format']) ? '.' : $settings['currency_format'];

        $this->add_render_attribute('button_text', 'class', [
            'elementor-price-table__button',
            'elementor-button',
            'elementor-size-' . $settings['button_size'],
        ]);

        if (!empty($settings['link']['url'])) {
            $this->add_render_attribute('button_text', 'href', $settings['link']['url']);

            if (!empty($settings['link']['is_external'])) {
                $this->add_render_attribute('button_text', 'target', '_blank');
            }
        }

        if (!empty($settings['button_hover_animation'])) {
            $this->add_render_attribute('button_text', 'class', 'elementor-animation-' . $settings['button_hover_animation']);
        }

        if ( !empty($settings['icon']) ) {
            $this->add_render_attribute( 'i', 'class', $settings['icon'] );
            $this->add_render_attribute( 'i', 'aria-hidden', 'true' );
        }

        $this->add_render_attribute('heading', 'class', 'elementor-price-table__heading');
        $this->add_render_attribute('description', 'class', 'elementor-price-table__heading-description');
        $this->add_render_attribute('period', 'class', 'elementor-price-table__period');
        $this->add_render_attribute('item_repeater', 'class', 'item-active');

        $this->add_inline_editing_attributes('heading', 'none');
        $this->add_inline_editing_attributes('description');
        $this->add_inline_editing_attributes('button_text');
        $this->add_inline_editing_attributes('item_repeater');

        $period_element = '<div ' . $this->get_render_attribute_string('period') . '>/' . $settings['period'] . '</div>';


        ?>

        <div class="elementor-price-table opal-box-clip-sm">
            <div class="elementor-price-table__wrapper-header">

                <?php
                $pricing_number = '';
                if(!empty($settings['price'])) {
                    $pricing_string = (string)$settings['price'];
                    $pricing_array = explode('.', $pricing_string);
                    if (isset($pricing_array[1]) && strlen($pricing_array[1]) < 2) {
                        $decimals = 1;
                    } else {
                        $decimals = 2;
                    }

                    if (count($pricing_array) < 2) {
                        $decimals = 0;
                    }

                    if (empty($settings['currency_format'])) {
                        $dec_point = '.';
                        $thousands_sep = ',';
                    } else {
                        $dec_point = ',';
                        $thousands_sep = '.';
                    }
                    $pricing_number = number_format($settings['price'], $decimals, $dec_point, $thousands_sep);
                }
                ?>
                <!--            icon box pricing-->
                <?php if ( !empty($settings['icon']) ):?>
                    <div class="elementor-price-table__icon-wrapper">
                <span class="elementor-icon">
                    <i <?php echo $this->get_render_attribute_string( 'i' ); ?>></i>
                    <span class="price-table-icon-bkg"></span>
                </span>
                    </div>
                <?php endif;?>
                <!--            end icon box-->

                <?php if ($settings['heading']) : ?>
                    <div class="elementor-price-table__header">
                        <?php if (!empty($settings['heading'])) : ?>
                            <h3 <?php echo $this->get_render_attribute_string('heading'); ?>><?php echo $settings['heading']; ?></h3>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>

                <?php if (!empty($settings['heading_description'])) : ?>
                    <div class="elementor-price-table__description">
                        <p <?php echo $this->get_render_attribute_string('description'); ?>><?php echo $settings['heading_description']; ?></p>
                    </div>
                <?php endif; ?>

                <div class="elementor-price-table__price">
                    <?php if (!empty($settings['price'])) : ?>
                        <?php if (!empty($symbol)) : ?>
                            <span class="elementor-price-table__currency"><?php echo $symbol; ?></span>
                        <?php endif; ?>
                        <span class="elementor-price-table__integer-part"><?php echo $pricing_number; ?></span>
                    <?php endif; ?>
                </div>

                <!--            html period-->
                <?php if (!empty($settings['period'])) : ?>
                    <?php echo $period_element; ?>
                <?php endif; ?>

                <?php if (!empty($settings['image']['url'])) { ?>
                    <?php
                    $image_html = Group_Control_Image_Size::get_attachment_image_html($settings, 'full', 'image');
                    echo $image_html;
                }
                ?>
            </div>

            <!--            end header-->

            <?php if (!empty($settings['features_list'])) : ?>
                <ul class="elementor-price-table__features-list">
                    <?php foreach ($settings['features_list'] as $index => $item) :
                        $repeater_setting_key = $this->get_repeater_setting_key('item_text', 'features_list', $index);
                        $this->add_inline_editing_attributes($repeater_setting_key);
                        ?>
                        <li class="elementor-repeater-item-<?php echo $item['_id']; ?>">
                            <?php if ($item['item_check']):?>
                                <?php $class_active = 'item_check';
                                $icon_item = '<i class="opal-icon-check" aria-hidden="true"></i>';
                            else: ?>
                                <?php $class_active = '';
                                $icon_item = '<i class="opal-icon-times" aria-hidden="true"></i>';

                            endif;?>
                            <div class="elementor-price-table__feature-inner <?php echo esc_attr__($class_active,'konstruktic-core');?>">
                                <?php echo $icon_item; ?>
                                <?php if (!empty($item['item_text'])) : ?>
                                    <span <?php echo $this->get_render_attribute_string($repeater_setting_key); ?>>
										<?php echo $item['item_text']; ?>
									</span>
                                <?php else :
                                    echo '&nbsp;';
                                endif;
                                ?>
                            </div>
                        </li>
                    <?php endforeach; ?>
                </ul>
            <?php endif; ?>

            <?php if (!empty($settings['button_text'])) : ?>
                <div class="elementor-price-table__footer">
                    <?php if (!empty($settings['button_text'])) : ?>
                        <a <?php echo $this->get_render_attribute_string('button_text'); ?>>
                            <span class="elementor-price-table_button-text"><?php echo $settings['button_text']; ?></span>
                            <span class="elementor-price-table_button-icon">
                                <i class="opal-icon-long-arrow-right" aria-hidden="true"></i>
                            </span>
                        </a>
                    <?php endif; ?>

                </div>
            <?php endif; ?>

            <?php if ( 'yes' === $settings['show_ribbon'] ) :
			$this->add_render_attribute( 'ribbon-wrapper', 'class', 'elementor-price-table__ribbon' );
			?>
            <div <?php echo $this->get_render_attribute_string( 'ribbon-wrapper' ); ?>>
            </div>
            <div class="elementor-price-table-icon">
                <i class="fa fa-star" aria-hidden="true"></i>
            </div>
		<?php endif; ?>

        </div>
        <?php
    }
}

$widgets_manager->register(new OSF_Elementor_Price_Table());